; Helper scripts for the Teamtalk application.
;
; Author:  Doug Lee

include "hjconst.jsh"
include "tt_jcpdict.jsm"

;========================== Module jcp (from jcpdict) ==========================
; Control property manager for JAWS using collections.
; Callback: jcp__setCustomProps.
; Set all control-specific properties there to make them work in speech and Braille consistently.
;
; Author:  Doug Lee

include "winstyles.jsh"
const
; Default cache lifetime for jcp__setAll() in milliseconds.
	jcp___DefaultCacheTicks = 500

globals
; Master on/off switch for the system, set by jcp__enable(), used for testing.
	int jcp___disabled,
; Cache lifetime in effect at any given time.
	int jcp___cacheTicks,
; True when this system is already active (used to avoid "unknown function call" messages)
	int jcp___inside,
; Last script key and name of script that was just run.
; Set by KeyPressedEvent (via jcp___setCaller) and used by jcp__getActionClass().
	string jcp___key,
	string jcp___caller,
; Caches for getObjectSubtypeCode() and jcp__getID(),
; because they can get called below enough times per second to slow things down.
; Example symptom: A one-second delay on a PCCursor() script call.
	collection jcp___intcache,
	int jcp___intTick

variant function jcp___vcast(variant v)
return v
endFunction

void function autoStartEvent()
jcp___cacheTicks = jcp___DefaultCacheTicks
jcp___intcache = new collection
endFunction

int function jcp__enable(int flag)
; Master on/off switch:  pass 0 to turn off, 1 for on, 2 to query without changing, and anything else to toggle.
; (This arrangement was chosen to make direct on/off make sense while making it easy to use isSameScript() to query on one press and toggle on more.)
if flag == 0 || flag == 1
	jcp___disabled = !flag
elif flag != 2
	jcp___disabled = !jcp___disabled
endIf
if jcp___disabled
	sayMessage(OT_No_Disable, "JCP disabled")
else
	sayMessage(OT_No_Disable, "JCP enabled")
endIf
return !jcp___disabled
endFunction

int function jcp__cache(int ticks)
; Set how long the cache of control properties can be kept before a forced refresh.
; Determines how often setCustomControlProps() is called.
; Pass 0 to use the internal default value.
; WARNING: Setting this too low can cause a major performance hit on systems with Braille displays,
; because setCustomControlProps() can be called over 40 times per second in such cases.
; Returns the old value that is being replaced.
; ticks: The maximum number of ticks (milliseconds) to keep the cache valid,
; or 0 to use the internal default value.
var
	int old
old = jcp___cacheTicks
if !ticks
	ticks = jcp___DefaultCacheTicks
endIf
jcp___cacheTicks = ticks
return old
endFunction

int function jcp__isEmpty(collection props)
; Returns True if props represents an empty property set.
var string k
forEach k in props
	; If this line is reached, there's at least one property.
	return False
endForEach
; And if this line is reached, there are no properties.
return True
endFunction

collection function jcp__new()
; Returns an empty props structure.
var collection c c = new collection
return c
endFunction

string function jcp__getActionClass()
; Get the type of action that appears to have caused the current jcpDict invocations:
;	tab:  Navigation among controls.
;	arrow:  Navigation within a control.
var
	string id
id = stringLower(jcp___key)
if stringContains(id, "tab")
	return "tab"
elif stringContains(id, "arrow")
|| stringContains(id, "home")
|| stringContains(id, "end")
	return "arrow"
endIf
return ""
endFunction

variant function jcp__get(collection props, string whichProp)
; Get the value of whichProp from props and return it.
; Case is not significant in property names.
; Dotted subclassing like fieldname.brl is permitted:
; If fieldname.brl exists it is returned, else if fieldname exists it is returned, else null is returned.
; Implemented subclassings:
;	- .brl for Braille calls.
;	- .<caller> for functions in this file that call jcp__get().
;	- .<class> for keys/scripts that initiated this jcp__get call (see jcp__getActionClass()).
; The latter of these is implemented internally here, not passed in whichProp.
; <caller> takes precedence over <class>.
; If a value begins with @, it is taken as a function call, and the return value of the call is returned.
var
	int pos,
	string val,
	string class,
	int done
val = ""
done = False
class = ""
while !done
	val = ""
	if collectionItemExists(props, whichProp)
		val = props[whichProp]
	endIf
	if val
		if stringLeft(val, 1) == "@"
			val = formatStringWithEmbeddedFunctions("<" +stringChopLeft(val, 1) +">")
		endIf
		return val
	endIf
	pos = stringContainsFromRight(whichProp, ".")
	if pos
		whichProp = stringLeft(whichProp, pos-1)
		if !whichProp
			done = True
		endIf
	elif !class
		class = jcp__getActionClass()
		if class
			whichProp = whichProp +"." +jcp__getActionClass()
		else
			done = True
		endIf
	else
		done = True
	endIf
endWhile
return ""
endFunction

int function jcp__prepend(collection props, string whichProp, string s)
; Prepend s to the property value indicated.
; whichProp should indicate a string property, not a numeric one.
var string val = jcp__get(props, whichProp)
if val
	val = s +" " +val
else
	val = s
endIf
jcp__set(props, whichProp, val)
endFunction

int function jcp__append(collection props, string whichProp, string s)
; Append s to the property value indicated.
; whichProp should indicate a string property, not a numeric one.
var string val = jcp__get(props, whichProp)
if val
	val = val +" " +s
else
	val = s
endIf
jcp__set(props, whichProp, val)
endFunction

void function jcp__setBits(collection props, string whichProp, int bits)
; Logically OR a bit or set of bits into a property value.
; whichProp should indicate a numeric property, not a string one.
var int val = jcp__get(props, whichProp) +0
val = val | bits
jcp__set(props, whichProp, val)
endFunction

void function jcp__clearBits(collection props, string whichProp, int bits)
; Logically AND a bit or set of bits out of a property value.
; whichProp should indicate a numeric property, not a string one.
var int val = jcp__get(props, whichProp) +0
val = val & (~bits)
jcp__set(props, whichProp, val)
endFunction

variant function jcp__setItemPos(collection props, variant pos, variant cnt)
; Set the Position property by its component parts.
; Also sets ItemPos and ItemCount.
; Recommended over setting Position or ItemPos/ItemCount directly.
; For convenience, stringToInt conversions are performed here if necessary
; so callers don't have to do it.
var int p = stringToInt(pos)
var int c = stringToInt(cnt)
jcp__set(props, "ItemPos", p)
jcp__set(props, "ItemCount", c)
var string buf
if p
	buf = formatString(jcp___MOfN, intToString(p), intToString(c))
else
	buf = formatString(jcp___NItems, intToString(c))
endIf
jcp__set(props, "Position", buf)
return True
endFunction

int function jcp__setTablePos(collection props, int rowIdx, int rowCount, int colIdx, int colCount)
; Set table row/column position and count properties.
; The "+0"'s avoid beeps when nulls are passed in.
return jcp__set(props, "row", rowIdx+0)
&& jcp__set(props, "rows", rowCount+0)
&& jcp__set(props, "col", colIdx+0)
&& jcp__set(props, "cols", colCount+0)
endFunction

string function jcp__getID(string sOrigin)
var
	int left, int right, int top, int bottom
if sOrigin == "current" || (sOrigin == "focus" && isPCCursor())
	if getObjectRect(left, right, top, bottom)
		return "obj" +intToString(left) +"," +intToString(top)
	else
		; Use the active cursor location.
		left = getCursorCol()
		top = getCursorRow()
		return "point" +intToString(left) +"," +intToString(top)
	endIf
elif sOrigin == "focus"
	; Focus but PC cursor not active; use the PC cursor location explicitly.
	saveCursor()  PCCursor()
	left = getCursorCol()
	top = getCursorRow()
	restoreCursor()
	return "point" +intToString(left) +"," +intToString(top)
else
	; sOrigin is not focus or current.  Use sOrigin itself as an ID.
	return sOrigin
endIf
endFunction

globals
	int jcp___lastTick,
	string jcp___lastID,
	string jcp___lastProps
int function jcp__setAll(collection byRef props, string sOrigin, /*?*/ int forceNoCache)
; Sets props to be a structure for other controlprop* functions.
; Obtains any properties of the indicated control that stray from what JAWS would normally discover and report.
; Returns True if anything was done or null if no special handling of this control is necessary.
var
	string id,
	int tc
; Start with a clean slate.
props = jcp__new()
if userBufferIsActive()
	return False
endIf
if !sOrigin
	sOrigin = "focus"
endIf

; Cache property sets (avoids serious CPU hits when a Braille display is active)
tc = getTickCount()
id = jcp___cachedVal(sOrigin)
if !forceNoCache && tc -jcp___lastTick <= jcp___cacheTicks
&& id && !stringCompare(jcp___lastID, id, True)
	props = jcp___vcast(jcp___lastProps)
	return !jcp__isEmpty(props)
endIf
jcp___lastID = jcp___vcast(id)
jcp___lastTick = tc

jcp___lastProps = jcp___vcast(props)

; Set any custom properties via the callback function.
;var int tcj = getTickCount()
jcp__setCustomProps(props, sOrigin)
;sayString(formatString("%2", id, intToString(getTickCount()-tcj)))
if !jcp__isEmpty(props)
	; Set the origin so jcp__say() will know where to get defaults.
	if !stringLength(jcp__get(props, "origin"))
		jcp__set(props, "origin", sOrigin)
	endIf
	jcp___lastProps = jcp___vcast(props)
	return True
endIf
return False
endFunction

int function jcp__setByOrigin(string sOrigin, string what, handle byRef hwnd, object byRef o, int byRef childID)
; Set origin info by reference.
; Allowed values for What:
;	h - Window handle.
;	m - MSAA object and childID
;	n - Nav ID (implies h).
;	c - Cursor-centric function support (e.g., getObjectName etc.).
; If passing c, first activate the cursor that will be used for getObject* etc. calls.
; At present, m and n are mutually exclusive.
; Returns True on success and False on failure.
what = stringLower(what)

; hwnd origin case.
if stringLeft(sOrigin, 4) == "hwnd"
	hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
	; Hwnd comes back whether requested or not here.
	if stringContains(what, "m")
		o = getObjectFromEvent(hwnd, -4, 0, childID)
	elif stringContains(what, "n")
		; ToDo: Probably insufficient.
		childID = jcp___vcast(navGetCurrentObjectID())
	endIf
	if stringContains(what, "c")
		if getCurrentWindow() != hwnd
			moveToWindow(hwnd)
			; TODO:  Verify that this returns the right window.
		endIf
	endIf
	return True
endIf

; Focus or current-when-current-is-focus cases.
if sOrigin == "focus" || (sOrigin == "current" && isPCCursor())
	hwnd = getFocus()
	if stringContains(what, "m")
		o = getFocusObject(childID)
		if !o
			return False
		endIf
	elif stringContains(what, "n")
		childID = jcp___vcast(navGetFocusObjectID())
	endIf
	if !isPCCursor()
		; Make GetObject*() calls apply to the right thing.
		saveCursor()  PCCursor()
	endIf
	return True
endIf

; Current (when distinct from focus) case.
if sOrigin == "current"
	hwnd = getCurrentWindow()
	if stringContains(what, "m")
		o = getObjectFromEvent(hwnd, -4, 0, childID)
		if !o
			return False
		endIf
	elif stringContains(what, "n")
		childID = jcp___vcast(navGetCurrentObjectID())
	endIf
	return True
endIf

outputDebugString(formatString("jcp__setCustomProps: Unsupported origin: %1", sOrigin))
return False
endFunction

void function jcp__reset()
; Force the next jcp__setAll to reload, not use the cache.
jcp___lastTick = 0
jcp___intTick = 0
endFunction

/*
void function jcp__setCustomProps(collection props, string sOrigin)
; Set any custom properties required per situation.
; Callers are expected to override this function.
; props is empty on entry and can be modified but should not be assigned to.
; Properties understood by default (property name case insignificant):
; Properties used by both Braille and speech:
;	Name, type, state
;	containerName, containerType
;	value, position, dlgText
;	level (for treeViews)
;	itemPos, itemCount (convenient way to compose position)
;	Hotkey (used in SayTutorialHelpHotKey)
;	subtypeCode (default type for speech and SayTutorialHelp and returned by BrailleCallbackObjectIdentify for this control)
;	attributes (CTRL_* constants from hjconst.jsh defining the control's states)
;	beforeField, afterField (text to speak/Braille before/after this field)
; Speech-only properties:
;	tutor, beforeTutor (tutor overrides the normal tutor message for the subtype)
; Braille-only properties:
;	dlgPageName, contextHelp, time
; Special properties internal to this system:
;	Mask:  Masks all special handling (use with caller name, e.g., jcp__set(props, "Mask.SayLine", True)).
;	Origin:  The origin to use to get default property values with getObject* functions (usually the sOrigin value used to create this property set)
; Note that SubtypeCode will set Type if Type is not set already; same for Attributes and State.
; sOrigin values that must be handled:
;	focus, current
;	subtype<subtype> (for BrailleAddObject* functions)
;	hwnd<handle>
; Other possible sOrigin values:
;	point<x>,<y>:  Like current but at a specific point
;	obj[<level>]  (level 0 is current)
;	hwnd<handle>/{nav<id>|sdm<id>|MSAA<id>}:  Nav* function ID, SDM ID, or MSAA ID within a window
;	point<x>,<y>:{hwnd|nav|SDM|MSAA}:  Like the extended hwnd syntax but from a specific point
endFunction
*/

variant function jcp___cachedVal(string which)
var int tc = getTickCount()
if jcp___intTick && tc -jcp___intTick < 250
	if collectionItemExists(jcp___intcache, which)
		return jcp___intcache[which]
	endIf
else
	collectionRemoveAll(jcp___intcache)
	jcp___intTick = tc
endIf
; The sought value does not exist now.
if which == "stc"
	jcp___intcache[which] = getObjectSubtypeCode()
else
	; An id request.
	jcp___intcache[which] = jcp__getID(which)
endIf
return jcp___intcache[which]
endFunction

int function jcp___sayControl(handle hwnd, string name, string type, string state, string containerName, string containerType, string value, string position, string dialogText)
var
	string allAttribs,
	int markupIncluded
allAttribs = name +type +state +containerName +containerType +value +position +dialogText
if !allAttribs
	return False
endIf
markupIncluded = (stringContains(stringLower(allAttribs), "<voice") > 0)
if !markupIncluded
	name = SMMReplaceSymbolsWithMarkup(name)
	type = SMMReplaceSymbolsWithMarkup(type)
	state = SMMReplaceSymbolsWithMarkup(state)
	containerName = SMMReplaceSymbolsWithMarkup(containerName)
	containerType = SMMReplaceSymbolsWithMarkup(containerType)
	value = SMMReplaceSymbolsWithMarkup(value)
	position = SMMReplaceSymbolsWithMarkup(position)
	dialogText = SMMReplaceSymbolsWithMarkup(dialogText)
endIf
if dialogText
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value, position, dialogText)
elif position
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value, position)
elif value
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value)
elif containerType
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType)
elif containerName
	sayControlExWithMarkup(hwnd, name, type, state, containerName)
elif state
	sayControlExWithMarkup(hwnd, name, type, state)
elif type
	sayControlExWithMarkup(hwnd, name, type)
else
	sayControlExWithMarkup(hwnd, name)
endIf
return True
endFunction

variant function jcp___dispatch(string func)
return formatStringWithEmbeddedFunctions("<" +func +">")
endFunction

void function jcp__say(collection props, string caller)
var
	string origin,
	string callerName,
	string actionClass,
	string beforeField, string afterField,
	string name, string type, string state, string containerName, string containerType, string value, string position, string dialogText,
	int rowIdx, int rowCount, int colIdx, int colCount,
	int typeCode, int attributes,
	handle hwnd
callerName = stringSegment(caller, "(", 1)
var int spelling = False
if callerName == "spellLine"
	callerName = "sayLine"
	spelling = True
endIf
if jcp__get(props, "Mask."+callerName)
	jcp___dispatch(caller)
	return
endIf
origin = jcp__get(props, "origin")
if origin == "focus"
	hwnd = getFocus()
elif origin == "current"
	hwnd = getCurrentWindow()
elif stringLeft(origin, 4) == "hwnd"
	hwnd = stringToHandle(stringChopLeft(origin, 4))
else
	beep()
	jcp___dispatch(caller)
	return
endIf

; Because of how they're used, null and unset string values are treated equal here.
; More care must be taken with numeric values though.
name = jcp__get(props, "name."+callerName)
type = jcp__get(props, "type."+callerName)
if !type
	typeCode = jcp__get(props, "subtypeCode."+callerName)
	; This one is treated the same if 0 or unset.
	if typeCode
		type = smmStripMarkup(smmGetStartMarkupForControlType(typeCode))
		if !type
			type = " "
		endIf
	endIf
endIf
state = jcp__get(props, "state."+callerName)
if !state
	attributes = jcp__get(props, "attributes."+callerName)
	; Attributes of 0 are supported explicitly here.
	if collectionItemExists(props, "attributes")
		if !typeCode
			typeCode = 1  ; all seem to work equally well
		endIf
		state = smmStripMarkup(smmGetStartMarkupForControlState(typeCode, attributes))
		if !state
			state = " "
		endIf
		if !type
			type = " "
		endIf
	endIf
endIf
containerName = jcp__get(props, "containerName."+callerName)
containerType = jcp__get(props, "containerType."+callerName)
value = jcp__get(props, "value."+callerName)
position = jcp__get(props, "position."+callerName)
dialogText = jcp__get(props, "DlgText."+callerName)
; beforeField is useful when assigning a name adversely changes other elements of how JAWS says a control.
beforeField = jcp__get(props, "beforeField."+callerName)
; afterField is useful for things like "minutes" in "Mark me away after _____ minutes"
afterField = jcp__get(props, "afterField."+callerName)

if stringLeft(origin, 4) == "hwnd"
	saveCursor() invisibleCursor() saveCursor()
	moveToWindow(hwnd)
endIf

actionClass = jcp__getActionClass()
	if caller != "sayLine" && actionClass != "arrow" && beforeField
		say(beforeField, OT_Control_Name, True)
	endIf

if caller == "sayLine" || actionClass == "arrow"
	containerName = ""
	containerType = ""
	dialogText = ""
endIf

if stringLength(name +type +state +containerName +containerType +value +position +dialogText)
	; Use SayControlEx or SayControlExWithMarkup.
	var int customName, int customType, int customState, int customValue, int customPosition
	customName = stringLength(name)
	if !name name = getObjectName() endIf
	customType = stringLength(type)
	if !type
		; Convert types that JAWS normally does not announce directly.
		var int tcode = getObjectSubtypeCode()
		var int tcode1 = 0
		if tcode == WT_ListItem && WT_ListItem == 86
			; WT_List == 85 but some non-English JAWS versions are
			; missing this constant in hjconst.jsh.
			tcode1 = 85
		elif tcode == WT_ListBoxItem
			tcode1 = WT_ListBox
		elif tcode == WT_ListViewItem
			tcode1 = WT_ListView
		elif tcode == WT_TreeViewItem
			tcode1 = WT_TreeView
		endIf
		if tcode1
			type = smmStripMarkup(smmGetStartMarkupForControlType(tcode1))
		else
			type = getObjectSubtype()
		endIf
	endIf
	customState = stringLength(state)
	if !state
		state = getObjectState()
		if stringCompare(stringStripAllBlanks(state), jcp___stateSelected, False) == 0
			; This normally doesn't happen, but JAWS doesn't say it normally anyway, so if it does show up, get rid of it.
			state = " "
		endIf
	endIf
	; No function for containerName/Type.
	customValue = stringLength(value)
	if !value value = getObjectValue() endIf
	customPosition = stringLength(position)
	if !position position = positionInGroup() endIf
	; TODO: No default dialog text here, could try getDialogStaticText().

	; Caller-specific adjustments.
	if caller == "sayLine" || actionClass == "arrow"
		containerName = " "
		dialogText = " "
		if !stringContains(":button:radiobutton:edit:readonlyedit:", ":"+stringLower(stringStripAllBlanks(type))+":")
			type = " "
		endIf
		if value
			name = " "
		endIf
		if caller != "sayLine"
			position = " "
		endIf
	elif stringStartsWith(caller, "sayObjectActiveItem")
		beforeField = ""
		type = " "
		containerName = " "
		dialogText = " "
		if stringContains(caller, "0")
			; Position info is not to be spoken.
			position = " "
		endIf
	endIf

	if spelling
		spellString(name +value)
	elif !jcp___sayControl(hwnd, name, type, state, containerName, containerType, value, position, dialogText)
		jcp___dispatch(caller)
	endIf
else
	jcp___dispatch(caller)
endIf  ; stringLength(name +type +state +containerName +containerType +value +position +dialogText)

; Indices of 0 and unset are treated equal.
rowIdx = jcp__get(props, "row."+callerName)
if caller != "sayLine" && rowIdx
	var string tblfmt
	rowCount = jcp__get(props, "rows."+callerName)
	colIdx = jcp__get(props, "col."+callerName)
	colCount = jcp__get(props, "cols."+callerName)
	tblfmt = "row %1"
	if rowCount
		tblfmt = tblfmt +" of %2"
	endIf
	if colIdx
		tblfmt = tblfmt +", column %3"
		if colCount
			tblfmt = tblfmt +" of %4"
		endIf
	endIf
	sayUsingVoice(VCTX_Message, formatString(tblfmt, intToString(rowIdx), intToString(rowCount), intToString(colIdx), intToString(colCount)), OT_Position)
endIf  ; rowIdx

if caller != "sayLine" && actionClass != "arrow" && afterField
	say(afterField, OT_Control_Name, True)
endIf
if caller != "sayLine" && actionClass != "arrow"
	; Tutor messages don't speak on SayLine, so beforeTutor doesn't either.
	var
		string beforeTutor
	beforeTutor = jcp__get(props, "beforeTutor")
	if !stringIsBlank(beforeTutor)
		; See JAWS 8's tutorialHelp.jss::sayTutorialHelp() for the rationale for using this SayUsingVoice call.
		; TODO: This may speak in a few undesirable places depending on user settings.
		sayUsingVoice(VCTX_Message, beforeTutor, OT_Line)
	endIf
endIf
endFunction

void function jcp___setCaller(int setting)
if setting
	jcp___key = getCurrentScriptKeyName()
	jcp___caller = getScriptAssignedTo(jcp___key)
	; Clear that shortly.
	scheduleFunction("jcp___setCaller", 5)
else
	jcp___key = ""
	jcp___caller = ""
endIf
endFunction

void function keyPressedEvent(int nKey, string strKeyName, int nIsBrailleKey, int nIsScriptKey)
; Reset the cache on scripts in case they change focus, values, etc.
if nIsBrailleKey || nIsScriptKey
	jcp__reset()
	jcp___setCaller(True)
else
	jcp___setCaller(False)
endIf
keyPressedEvent(nKey, strKeyName, nIsBrailleKey, nIsScriptKey)
endFunction

void function sayObjectTypeAndText(int nLevel)
; An override to improve handling of some windows.
var
	collection props,
	handle hwnd,
	string name,
	string buf
if jcp___disabled || jcp___inside || !isPCCursor() || nLevel > 0
	; System disabled, recursive call, non-PC cursor situation, or JAWS 8+ call on non-current control; handle normally.
	sayObjectTypeAndText(nLevel)
	return
endIf

; Look for special handling of the current control.
if jcp__setAll(props, "current", True)
	jcp___inside = True
	jcp__say(props, "sayObjectTypeAndText(" +intToString(nLevel) +")")
	jcp___inside = False
	return
endIf

sayObjectTypeAndText(nLevel)
endFunction

void function sayWindowTypeAndText(handle hwnd)
; Look for special handling of the indicated window.
var
	collection props,
	string sOrigin
if jcp___disabled || jcp___inside
	sayWindowTypeAndText(hwnd)
	return
endIf

if hwnd == getFocus()
	sOrigin = "focus"
elif hwnd == getCurrentWindow()
	sOrigin = "current"
else
	sOrigin = "hwnd"+intToString(hwnd)
endIf
if !jcp___inside && jcp__setAll(props, sOrigin, True)
	jcp___inside = True
	jcp__say(props, "sayWindowTypeAndText(" +intToString(hwnd) +")")
	jcp___inside = False
	return
endIf
sayWindowTypeAndText(hwnd)
endFunction

void function sayLine(int iDrawHighlights, int bSayingLineAfterMovement)
; Look for special handling of the current line.
var
	collection props,
	string sOrigin
if jcp___disabled || jcp___inside || !isPCCursor()
	sayLine(iDrawHighlights, bSayingLineAfterMovement)
	return
endIf
sOrigin = "current"
if !jcp__setAll(props, sOrigin, True)
	sayLine(iDrawHighlights, bSayingLineAfterMovement)
	return
endIf
var int set_stc = jcp__get(props, "SubtypeCode")
var int isMLE = False
if set_stc && set_stc != WT_Multiline_Edit
	isMLE = False
elif getObjectSubtypeCode() == WT_MultiLine_Edit
; Catches controls that are being specifically set to this type.
|| set_stc == WT_MultiLine_Edit
; Catches controls that JAWS doesn't know are multiline edits.
; The second condition catches read-only multiline edits, which getObjectSubtypeCode() just calls WT_ReadOnlyEdit.
; the SendMessage checks for if the control accepts EM_SetSel, which implies that it's actually an edit control.
; 0x87 is WM_GetDlgCode, and 8 is DLGC_HasSetSel.
|| ((getWindowStyleBits(getCurrentWindow()) & ES_Multiline)
	&& (sendMessage(getCurrentWindow(), 0x87, 0, 0) & 8))
		isMLE = True
endIf
if isMLE
	; Avoid special processing on up/down arrows.
	jcp__set(props, "Mask.arrow", True)
	jcp__set(props, "Mask.sayLine", True)
	; Don't say a control name for SayLine in a multiline edit when it contains something.
	if getObjectValue()
	; or WM_GetTextLength returns non-zero.
	|| (sendMessage(getCurrentWindow(), 0x0E, 0, 0) > 0)
		jcp__set(props, "Name", " ")
	endIf
endIf
jcp___inside = True
jcp__say(props, "sayLine(" +intToString(iDrawHighlights) +", " +intToString(bSayingLineAfterMovement) +")")
jcp___inside = False
endFunction

void function spellLine()
; Look for special handling of the current line.
var
	collection props,
	string sOrigin
if jcp___disabled || jcp___inside || !isPCCursor()
	spellLine()
	return
endIf
sOrigin = "current"
if !jcp__setAll(props, sOrigin, True)
	spellLine()
	return
endIf
var int set_stc = jcp__get(props, "SubtypeCode")
var int isMLE = False
if set_stc && set_stc != WT_Multiline_Edit
	isMLE = False
elif getObjectSubtypeCode() == WT_MultiLine_Edit
; Catches controls that are being specifically set to this type.
|| set_stc == WT_MultiLine_Edit
; Catches controls that JAWS doesn't know are multiline edits.
; The second condition catches read-only multiline edits, which getObjectSubtypeCode() just calls WT_ReadOnlyEdit.
; the SendMessage checks for if the control accepts EM_SetSel, which implies that it's actually an edit control.
; 0x87 is WM_GetDlgCode, and 8 is DLGC_HasSetSel.
|| ((getWindowStyleBits(getCurrentWindow()) & ES_Multiline)
	&& (sendMessage(getCurrentWindow(), 0x87, 0, 0) & 8))
		isMLE = True
endIf
if isMLE
	; Avoid special processing on up/down arrows.
	jcp__set(props, "Mask.arrow", True)
	jcp__set(props, "Mask.sayLine", True)
	; Don't say a control name for SayLine in a multiline edit when it contains something.
	if getObjectValue()
	; or WM_GetTextLength returns non-zero.
	|| (sendMessage(getCurrentWindow(), 0x0E, 0, 0) > 0)
		jcp__set(props, "Name", " ")
	endIf
endIf
jcp___inside = True
jcp__say(props, "spellLine()")
jcp___inside = False
endFunction

void function sayObjectActiveItem(optional int sayPositionInfo)
; Look for special handling of the current item.
var
	collection props,
	string sOrigin
if jcp___disabled || jcp___inside
	sayObjectActiveItem(sayPositionInfo)
	return
elif getObjectSubtypeCode() == WT_MultiLine_Edit
; The second condition catches read-only multiline edits, which getObjectSubtypeCode() just calls WT_ReadOnlyEdit.
; the SendMessage checks for if the control accepts EM_SetSel, which implies that it's actually an edit control.
; 0x87 is WM_GetDlgCode, and 8 is DLGC_HasSetSel.
|| ((getWindowStyleBits(getCurrentWindow()) & ES_Multiline) && (sendMessage(getCurrentWindow(), 0x87, 0, 0) & 8))
	sayObjectActiveItem(sayPositionInfo)
	return
endIf

sOrigin = "current"
if isPCCursor() && jcp__setAll(props, sOrigin, True)
	jcp___inside = True
	jcp__say(props, "sayObjectActiveItem(" +intToString(sayPositionInfo) +")")
	jcp___inside = False
	return
endIf
sayObjectActiveItem(sayPositionInfo)
endFunction

int function getTreeViewLevel()
if jcp___disabled || jcp___inside || !isPCCursor()
	return getTreeViewLevel()
endIf
var collection props
if jcp__setAll(props, "current", False)
	; Level 0 explicitly supported here.
	if collectionItemExists(props, "level")
		return stringToInt(jcp__get(props, "Level"))
	endIf
endIf
return getTreeViewLevel()
endFunction

void function sayTreeViewItem()
if jcp___disabled || jcp___inside || !isPCCursor()
	sayTreeViewItem()
	return
endIf
var collection props
if jcp__setAll(props, "current", False)
	sayObjectActiveItem(False)
	return
endIf
sayTreeViewItem()
endFunction

int function sayTutorialHelp(int iSubType, int isScriptKey)
; Blocks tutor messages when set to blank.
; Also allows iSubtype to be overridden by a subtypeCode property in props.
; Finally allows the subtype to be obtained from a different origin (a specific window).
var
	collection props,
	string msg
if jcp___disabled || jcp___inside || !isPCCursor() || getObjectSubtypeCode() != iSubtype
	; We know nothing of this, so let JAWS figure it out.
	return sayTutorialHelp(iSubType, isScriptKey)
endIf
if jcp__setAll(props, "current", False)
	msg = jcp__get(props, "tutor")
	if stringLength(msg) > 0 && stringIsBlank(msg)
		; Somebody said say nothing, so say nothing.
		return True  ; but say we said what we needed to say.
	elif jcp__getActionClass() == "arrow"
		; No tutor text on arrows.
		return True
	endIf
	var int iSubtype1
	iSubtype1 = jcp__get(props, "subtypeCode")
	; subtypeCode of 0 explicitly allowed here.
	if collectionItemExists(props, "subtypeCode")
		iSubtype = iSubtype1
	else
		; This is just for when jcp__setCustomProps() changes the origin.
		var string sOrigin, handle hwnd
		sOrigin = jcp__get(props, "origin")
		if stringLeft(sOrigin, 4) == "hwnd"
			hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
			saveCursor() invisibleCursor() saveCursor()
			moveToWindow(hwnd)
			iSubtype = getObjectSubtypeCode()
		endIf  ; sOrigin == "hwnd"
	endIf  ; iSubtype1, else
endIf  ; jcp__setAll got something
return sayTutorialHelp(iSubType, isScriptKey)
endFunction

int function sayTutorialHelpHotKey(handle hwnd, int isScriptKey)
var
	collection props,
	string msg
if jcp___disabled || jcp___inside || !isPCCursor() || hwnd != getFocus()
	; We know nothing of this, so let JAWS figure it out.
	return sayTutorialHelpHotKey(hwnd, isScriptKey)
endIf
if jcp__setAll(props, "current", False)
	msg = jcp__get(props, "hotkey")
	if stringLength(msg) > 0
		if stringIsBlank(msg)
			; Somebody said say nothing, so say nothing.
		elif jcp__getActionClass() == "arrow"
			; No tutor text on arrows.
		else
			sayUsingVoice(VCTX_Message, msg, OT_Access_Key)
		endIf
		return True
	endIf
	; This is just for when jcp__setCustomProps() changes the origin.
	var string sOrigin
	sOrigin = jcp__get(props, "origin")
	if stringLeft(sOrigin, 4) == "hwnd"
		hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
		saveCursor() invisibleCursor() saveCursor()
		moveToWindow(hwnd)
	endIf
endIf
sayTutorialHelpHotKey(hwnd, isScriptKey)
endFunction

string function getCustomTutorMessage()
; Called in JAWS 7.1+ to get custom tutor messages.
var
	collection props,
	string msg
msg = ""
if jcp___disabled || jcp___inside || !jcp__setAll(props, "current", False)
	return getCustomTutorMessage()
endIf
msg = jcp__get(props, "tutor")
if msg
	return msg
endIf
return getCustomTutorMessage()
endFunction

int function jcp___BrailleAddObjectHelper0(string whichProp, int nSubtype)
; Logic for BrailleAddObject* functions to use (wrapped by jcp___BrailleAddObjectHelper though).
var
	string sOrigin,
	collection props,
	int attribs,
	int x, int y,
	string val
if jcp___disabled
	return False
endIf
var int scode = jcp___cachedVal("stc")
if scode && scode != nSubtype
	; BrailleAddObject* function(s) called on a parent dialog control.
	var int level, int found
	level = 1
	found = False
	while !found && level <= 5
		if getObjectSubTypeCode(False, level) == nSubtype
			sOrigin = "focus" +intToString(level)
			found = True
		endIf
		level = level +1
	endWhile
	if !found
		; TODO:  This may cause some incorrect behavior, but it is hard to know when this should or shouldn't be done.
		; Example:  A .Net edit combo/spin box has been seen to return Spinbox for Braille and Edit Combo for speech.
		; GetObjectSubtypeCode returns ReadOnly Edit at level 0 and Combobox at level 1.
		;sOrigin = "subtype" +intToString(nSubtype)
		sOrigin = "focus"
	endIf
else
	sOrigin = "focus"
endIf
if !jcp__setAll(props, sOrigin, False)
	return False
endIf
if whichProp == "name"
	; Indices of 0 and unset are treated equal.
	val = jcp__get(props, "row.brl")
	if val
		var variant val1
		val1 = jcp__get(props, "col.brl")
		if val1
			val = formatString("r%1c%2", val, val1)
		else
			val = formatString("r%1", val)
		endIf
		BrailleAddString(val, 0,0, 0)
	endIf
	val = jcp__get(props, "beforeField.brl")
	if val
		BrailleAddString(val, 0,0, 0)
	endIf
endIf
val = jcp__get(props, whichProp +".brl")
if !val
	if whichProp == "type"
		scode = jcp__get(props, "subtypeCode.brl")
		; This one is treated the same if 0 or unset.
		if scode
			val = BrailleGetSubtypeString(scode)
			if !val
				val = " "
			endIf
		endIf
	elif whichProp == "state"
		attribs =jcp__get(props, "attributes")
		; Attributes of 0 are supported explicitly here.
		if collectionItemExists(props, "attributes")
			val = BrailleGetStateString (attribs)
			if !val
				val = " "
			endIf
		endIf
	endIf
endIf
var int retval = False
if val
	if !stringIsBlank(val)
		x = 0
		y = 0
		; "name" left out deliberately; prevents cursor from being placed on label instead of value in edit fields
		if whichProp == "type" || whichProp == "state" || whichProp == "value"
			x = getCursorCol()
			y = getCursorRow()
		endIf
		;if whichProp == "type" sayString(val) endIf
		BrailleAddString(val, x, y, 0)
	endIf  ; !stringIsBlank(val)
	retval = True
endIf  ; val
if whichProp == "value"
	val = jcp__get(props, "afterField.brl")
	if val
		if !retval
			; No custom value, but put any non-custom one here.
			; The BrailleAddObjectValue in this file should be our caller and so shouldn't run again.
			BrailleAddObjectValue(nSubtype)
		endIf
		BrailleAddString(val, 0,0, 0)
	endIf
endIf
return retval
endFunction

int function jcp___BrailleAddObjectHelper(string whichProp, int nSubtype)
; Called by all BrailleAddObject* functions.
var
	int retval
retval = jcp___BrailleAddObjectHelper0(whichProp, nSubtype)
if retval
	return retval
endIf
return jcp___dispatch("BrailleAddObject" +whichProp +"(" +intToString(nSubtype) +")")
endFunction

; These are all the BrailleAddObject* functions internally recognized as of JAWS 8.0.
int function BrailleAddObjectName(int nSubtype)
return jcp___BrailleAddObjectHelper("Name", nSubtype)
endFunction
int function BrailleAddObjectType(int nSubtype)
return jcp___BrailleAddObjectHelper("Type", nSubtype)
endFunction
int function BrailleAddObjectState(int nSubtype)
return jcp___BrailleAddObjectHelper("State", nSubtype)
endFunction
int function BrailleAddObjectValue(int nSubtype)
return jcp___BrailleAddObjectHelper("Value", nSubtype)
endFunction
int function BrailleAddObjectContainerName(int nSubtype)
return jcp___BrailleAddObjectHelper("ContainerName", nSubtype)
endFunction
int function BrailleAddObjectContainerType(int nSubtype)
return jcp___BrailleAddObjectHelper("ContainerType", nSubtype)
endFunction
int function BrailleAddObjectPosition(int nSubtype)
return jcp___BrailleAddObjectHelper("Position", nSubtype)
endFunction
int function BrailleAddObjectDlgPageName(int nSubtype)
return jcp___BrailleAddObjectHelper("DlgPageName", nSubtype)
endFunction
int function BrailleAddObjectDlgText(int nSubtype)
return jcp___BrailleAddObjectHelper("DlgText", nSubtype)
endFunction
int function BrailleAddObjectContextHelp(int nSubtype)
return jcp___BrailleAddObjectHelper("ContextHelp", nSubtype)
endFunction
int function BrailleAddObjectTime(int nSubtype)
return jcp___BrailleAddObjectHelper("Time", nSubtype)
endFunction
int function BrailleAddObjectLevel(int nSubtype)
return jcp___BrailleAddObjectHelper("Level", nSubtype)
endFunction

int function BrailleCallbackObjectIdentify()
; Allows WT_Unknown to be translated to jcp__get(props, "subtypeCode.brl").
; Set subtypeCode.brl to adjust the Brailled type and the set of BrailleAddObject* functions called for a control.
; Also allows the subtype to be obtained from a different origin (a specific window).
var int tc = getTickCount()
var
	collection props,
	int scode
if !jcp___disabled && !jcp___inside && jcp__setAll(props, "current", False)
	scode = jcp__get(props, "subtypeCode.brl")
	; subtypeCode of 0 explicitly allowed here.
	if collectionItemExists(props, "subtypeCode")
		return scode
	else
		; This is just for when jcp__setCustomProps() changes the origin.
		var string sOrigin, handle hwnd
		sOrigin = jcp__get(props, "origin")
		if stringLeft(sOrigin, 4) == "hwnd"
			hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
			saveCursor() invisibleCursor() saveCursor()
			moveToWindow(hwnd)
			scode = getObjectSubtypeCode()
			return scode
		endIf  ; sOrigin == "hwnd"
	endIf
endIf
return BrailleCallbackObjectIdentify()
endFunction

int function jcp__set(collection props, string whichProp, variant val, /*?*/ int allowFunctionCall)
; Set a control property in props.
; Case is not significant in property names.
; If allowFunctionCall is False (default) and val begins with "@",
; a space is prepended to val to keep jcp__get() from thinking it's a function call.
; Special itemPos/Count for setting parts of Position conveniently.
if whichProp == "itemPos" || whichProp == "itemCount"
	var string posdata, int pos
	posdata = jcp__get(props, "Position")
	if !posdata
		posdata = positionInGroup()
		if !posdata
			posdata = jcp___MOfN
		endIf
	endIf
	if whichProp == "itemPos"
		pos = stringContains(posdata, " ")
		posdata = stringChopLeft(" " +intToString(val) +stringChopLeft(posdata, pos-1), 1)
	else
		pos = stringContainsFromRight(posdata, " ")
		posdata = stringChopLeft(" " +stringLeft(posdata, pos) +intToString(val), 1)
	endIf
	whichProp = "position"
	val = jcp___vcast(posdata)
endIf
if !allowFunctionCall && stringLeft(val, 1) == "@"
	val = jcp___vcast(" " +val)
endIf
props[whichProp] = val
return True
endFunction

;======================== End module jcp (from jcpdict) ========================

