; Scripts for TeamTalk Classic
;
; Author:  Doug Lee
include "hjconst.jsh"
include "MSAAConst.jsh"
include "tt_classic.jsm"

use "tt_jcpdict.jsb"

const
; Script revision number, spoken by JAWSKey+Q.
	Script_Revision = "153"

script scriptFileName()
scriptAndAppNames(formatString(msgScriptName, Script_Revision))
endScript

void function autoStartEvent()
chat4__registerAllChatWindows()
; Skip all messages that arrived in any window since last TeamTalk focus.
chat4__speakNewText(0, 0, 0)
toolbar__watchForChanges(10)
endFunction

script sayNextWord()
if !_moveByUnit("nextWord")
	performScript sayNextWord()
endIf
endScript

script sayPriorWord()
if !_moveByUnit("priorWord")
	performScript sayPriorWord()
endIf
endScript

int function _moveByUnit(string func)
if !isPCCursor() || util__isSpecialFocus()
	return False
elif getObjectTypeCode() != WT_Edit
	return False
endIf
var collection rec = chat4__record()
if !rec || rec.type != "c"
	; Private chat windows work fine already (4.3 Classic at least).
	return False
endIf

; RouteInvisibleToPC fails here by leaving the invisible cursor some 65,000
; pixels off the bottom right of the screen.  RouteJAWSToPC comes closer but
; can still end up at the wrong end of a word. The MSAA caret is available
; even in the read-only channel message window and is accurately located,
; so we use that. [DGL, 2012-03-17, JAWS 12.0.1170]

var int left, int right, int top, int bottom, int xc, int yc
; -8 is ObjID_Caret.
if !MSAA__windowRect(getFocus(), -8, left, right, top, bottom)
	return False
endIf
xc = (left+right) /2
yc = (top+bottom) /2

saveCursor()  invisibleCursor()  saveCursor()
moveTo(xc, yc)
var int oldR = getRestriction()
setRestriction(RestrictWindow)
formatStringWithEmbeddedFunctions("<" +func +">")
setRestriction(oldR)
routePCToInvisible()
PCCursor()
func = stringLower(func)
func = stringReplaceSubstrings(func, "next", "")
func = stringReplaceSubstrings(func, "prior", "")
pause()
formatStringWithEmbeddedFunctions("<say" +func +">")
return True
endFunction

int function ReadMsg(string which, int isReviewSystem)
; Which should be +n (n forward), -n (n backward), n (message #n), "L" (last), or "C" (current).
var
	int msgno,
	int messageCount,
	int isJoinOrLeave,
	string buf
var collection rec = chat4__record()
msgno = rec.reviewMessageNumber
messageCount = chat4__messageCount()
var string ch
; Adjustment for reversed windows.
if rec.reversed
	ch = stringLeft(which, 1)
	var string rest = stringChopLeft(which, 1)
	if ch == "+"
		which = formatString("-%1", rest)
	elif ch == "-"
		which = formatString("+%1", rest)
	elif ch == "l"
		which = "1"
	elif ch == "c"
		; No change.
	else  ; 0-9, absolute position.
		which = intToString( messageCount -stringToInt(which) +1)
	endIf
endIf

; First deal with room switches.
if messageCount < 0
	sayMessage(OT_Error, msgMessageError)
	return False
endIf
if !messageCount
	msgno = 0
	sayMessage(OT_User_Requested_Information, msgNoMessages)
	if isReviewSystem
		rec.reviewMessageNumber = msgno
	endIf
	return False
elif msgno < 1
	msgno = 1
elif msgno > messageCount
	msgno = messageCount
endIf

ch = stringLeft(which, 1)
if ch == "L"
	msgno = messageCount
elif ch == "C"
	; Do nothing; leave msgno as is.
elif stringContains("+-", stringLeft(which, 1))
	msgno = msgno +stringToInt(which)
else
	msgno = stringToInt(which)
endIf

; Say the message after possible translation for welcome/goodbye style.
buf = chat4__messageText(msgno)
applyChatTimePreference(buf)
isJoinOrLeave = False

; the isReviewSystem part allows manually-requested messages to speak under all conditions.
var int chCycler = 0
if isReviewSystem || !ChCycler || (ChCycler == 1 && isJoinOrLeave) || (ChCycler == 2 && !isJoinOrLeave) || (ChCycler == 3 && isJoinOrLeave)
	if ch == "c"
		var string buf1
		; For reversed windows we have to flip the message number.
		var int mn1 = rec.reviewMessageNumber
		if rec.reversed
			mn1 = messageCount -mn1 +1
		endIf
		buf1 = formatString(msgMessagePos, intToString(mn1), intToString(messageCount))
		buf = formatString("%1: %2", buf1, buf)
	endIf
	sayMessage(OT_User_Requested_Information, buf)
elif !isReviewSystem && !isJoinOrLeave && (ChCycler == 3 || ChCycler == 4)
	scheduleFunction("PlayMessageSound", 2)
endIf

if msgno < 1
	msgno = 1
elif msgno > messageCount
	msgno = messageCount
endIf
if isReviewSystem
	rec.reviewMessageNumber = msgno
endIf
return True
endFunction

int function applyChatTimePreference(string byRef buf)
; Strip out the message time if requested by the user option for this.
if opt__get("chatMessageTimes")
	return False
endIf
var int pos = stringContains(buf, " ")
var string maybeTime = stringLeft(buf, pos-1)
if stringContains(maybeTime, ":")
&& stringContains("0123456789", stringLeft(maybeTime, 1))
&& stringContains("0123456789", stringRight(maybeTime, 1))
	buf = stringChopLeft(buf, pos)
	return True
endIf
return False
endFunction

Script ReadFirstMessage()
ReadMsg("1", True)
EndScript
Script ReadLastMessage()
ReadMsg("L", True)
EndScript
Script ReadNextMessage()
ReadMsg("+1", True)
EndScript
Script ReadPriorMessage()
ReadMsg("-1", True)
EndScript
Script ReadCurrentMessage()
var collection rec = chat4__record()
var string buf = chat4__messageText(rec.reviewMessageNumber)
applyChatTimePreference(buf)
if isSameScript()
	spellString(buf)
else
	ReadMsg("c", True)
endIf
EndScript
script readAbsoluteMessage(int idx)
if idx < 0
	idx = chat4__messageCount() +idx +1
endIf
readMsg(intToString(idx), False)
endScript

script topOfFile()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("first", False)
	return
endIf
performScript topOfFile()
endScript

script selectFromTop()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("first", True)
	return
endIf
performScript selectFromTop()
endScript

script bottomOfFile()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("last", False)
	return
endIf
performScript bottomOfFile()
endScript

script selectToBottom()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("last", True)
	return
endIf
performScript selectToBottom()
endScript

script ControlUpArrow()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("prev", False)
	return
endIf
performScript ControlUpArrow()
endScript

script prevByAge()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("prev", True)
	return
endIf
sayCurrentScriptKeyLabel()
typeCurrentScriptKey()
endScript

script controlDownArrow()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("next", False)
	return
endIf
performScript controlDownArrow()
endScript

script nextByAge()
if !util__isSpecialFocus() && isPCCursor() && getFocus() && getFocus() == roomtree__currentTreeWindow()
	roomtree__move("next", True)
	return
endIf
sayCurrentScriptKeyLabel()
typeCurrentScriptKey()
endScript

script nextChat()
; Move to the next TeamTalk window containing a visible chat.
if !chat4__focusNext(1)
	beep()
endIf
endScript

script prevChat()
; Move to the previous TeamTalk window containing a visible chat.
if !chat4__focusNext(-1)
	beep()
endIf
endScript

script sayStatuses()
; Speak the statuses of toolbar icons.
; On one press, provide an abbreviated indication.
; On a double press, list all statuses completely.
toolbar__sayStatuses(isSameScript())
endScript

script DoEnter()
; Makes Enter join a room if focus is on a room line or on someone within one.
if util__isSpecialFocus()
	typeCurrentScriptKey()
	sayCurrentScriptKeyLabel()
	return
endIf
var handle hwnd = getFocus()
if getWindowClass(hwnd) == "SysTreeview32"
	if _DoubleClick(hwnd)
		return
	endIf
	beep()
endIf
typeCurrentScriptKey()
sayCurrentScriptKeyLabel()
endScript

int function _DoubleClick(handle hwnd)
; Double-click the focused tree item.
; For a room node, this joins the room if "Double click enters a room" is set in TeamTalk.
; For a member, this opens a chat window.
; Returns True on success and False on error.
var object o = MSAA__objectFromWindow(hwnd)
if !o
	return False
endIf
var int childID = o.accFocus +0
if !childID
	; Even if this means focus is just on the tree itself,
	; that's not a room to join or a member to talk to.
	return False
endIf
; Even though we're using winclick to avoid having to mess with the mouse pointer,
; we probably have to have the node actually on screen.
; TODO: Could improve this to allow for partial visibility.
var int accState = o.accState(childID)
if (accState & (State_System_Invisible | State_System_OffScreen))
	return False
endIf
; Click the center of the node.
var int left, int right, int top, int bottom
if !MSAA__rect(o, childID, left, right, top, bottom)
	return False
endIf
var int x = (left +right) /2
var int y = (top +bottom) /2
winclick__post(hwnd, "LeftDblClick", x, y)
; Send seems to return 0 even on success.
return True
endFunction

script sayBottomLineOfWindow()
; Make JAWSKey+PgDn work even when the window is not visible.
if !util__isSpecialFocus()
	var string txt = ttutil__statusText(getFocus())
	if txt
		sayMessage(OT_Status, txt)
		return
	endIf
endIf
performScript sayBottomLineOfWindow()
endScript

void function jcp__setCustomProps(variant byRef props, string sOrigin)
; Set any custom properties required per situation.
; props is empty on entry, can be modified, and must be taken by reference.
; Properties understood by default (property name case insignificant):
; Properties used by both Braille and speech:
;	Name, type, state
;	containerName, containerType
;	value, position, dlgText
;	itemPos, itemCount (convenient way to compose position)
;	Hotkey (used in SayTutorialHelpHotKey)
;	subtypeCode (default type for speech and SayTutorialHelp and returned by BrailleCallbackObjectIdentify for this control)
;	attributes (CTRL_* constants from hjconst.jsh defining the control's states)
;	beforeField, afterField (text to speak/Braille before/after this field)
; Speech-only properties:
;	tutor, beforeTutor (tutor overrides the normal tutor message for the subtype)
; Braille-only properties:
;	dlgPageName, contextHelp, time, level
; Special properties internal to this system:
;	Mask:  Masks all special handling (use with caller name, e.g., set(props, "Mask.SayLine", True)).
;	Origin:  The origin to use to get default property values with getObject* functions (usually the sOrigin value used to create this property set)
; Note that SubtypeCode will set Type if Type is not set already; same for Attributes and State.
; sOrigin values that must be handled:
;	focus, current
;	subtype<subtype> (for BrailleAddObject* functions)
;	hwnd<handle>
; Other possible sOrigin values:
;	point<x>,<y>:  Like current but at a specific point
;	obj[<level>]  (level 0 is current)
;	hwnd<handle>/{nav<id>|sdm<id>|MSAA<id>}:  Nav* function ID, SDM ID, or MSAA ID within a window
;	point<x>,<y>:{hwnd|nav|SDM|MSAA}:  Like the extended hwnd syntax but from a specific point
var handle hwnd, object o, int childID
if isVirtualPCCursor()
	return
endIf
jcp__setByOrigin(sOrigin, "hm", hwnd, o, childID)
if util__isSpecialFocus()
	return
endIf
if ttutil__isMainWindow(getTopLevelWindow(hwnd))
	_labelMain(props, hwnd)
endIf
endFunction

int function _labelMain(variant byRef props, handle hwnd)
; Label the sliders and other items in the main window.
var string wclass = getWindowClass(hwnd)
var string name
if wclass == "Edit"
	jcp__set(props, "Name", msgChannelMessageInput)
	return True
elif wclass == "RichEdit20W"
	jcp__set(props, "Name", msgChannelMessageHistory)
	return True
elif wclass == "SysTreeView32"
	name = getObjectValue()
	; That includes flags because of a getObjectValue() override in this file.
	if !name
		name = " "
	endIf
	jcp__set(props, "Value", name)
	return True
endIf
if getWindowTypeCode(hwnd) != WT_Slider
	return False
endIf
var handle h1 = getFirstWindow(hwnd)
var int i = 0
var string names = msgSliderNames
while h1
	if getWindowTypeCode(h1) == WT_Slider
		i = i +1
	endIf
	if h1 == hwnd
		name = stringSegment(names, "|", i)
		jcp__set(props, "Name", name)
		return True
	endIf
	h1 = getNextWindow(h1)
endWhile
return False
endFunction

string function getObjectValue(/*?*/ int alwaysUseMSAA, int nLevel)
var string val = getObjectValue(alwaysUseMSAA, nLevel)
if nLevel != 0
	return val
endIf
if !util__isSpecialFocus() && isPCCursor()
&& getWindowClass(getFocus()) == "SysTreeView32"
	var string flags = roomtree__iconStatus()
	var string buf = ""
	if stringContains(flags, "s")
		buf = buf +" " +msgSending
	endIf
	if stringContains(flags, "i")
		buf = buf +" " +msgIdle
	endIf
	if stringContains(flags, "o")
		buf = buf +" " +msgOnline
	endIf
	val = stringTrimTrailingBlanks(val)
	if val && !getObjectName(True)
		val = ""
	endIf
	buf = stringChopLeft(buf, 1)
	val = string__join(val, " - ", buf)
endIf
return val
endFunction

void function ValueChangedEvent(handle hwnd, int objId, int childId, int nObjType, string sObjName, string sObjValue, /*?*/ int bIsFocusObject)
ValueChangedEvent(hwnd, objId, childId, nObjType, sObjName, sObjValue, bIsFocusObject)
if !ttutil__mainWindow(hwnd, True)
|| nObjType != WT_Edit
|| getWindowSubtypeCode(hwnd) != WT_ReadOnlyEdit
	return
endIf
var int sntStyle = opt__get("SpeakChatMessages")
	; Two ValueChangedEvents seem to fire on (at least channel) chat windows
	; when a message arrives. We therefore ask the speech to be
	; scheduled rather than immediate. [DGL, 2012-10-21]
chat4__speakNewText(hwnd, sntStyle, 3)
endFunction

globals int _sayingTitle
script sayWindowTitle()
_sayingTitle = True
performScript sayWindowTitle()
_sayingTitle = False
endScript

string function getWindowName(handle hwnd)
var string name = getWindowName(hwnd)
if _sayingTitle
&& hwnd == getAppMainWindow(getFocus())
	var string server = roomtree__serverName(hwnd)
	if server
		name = server +", " +name
	endIf
endIf
return name
endFunction


;================================= Module chat4 ================================
; ChatOutput window interface for TeamTalk Classic
; This is an iTextDocument interface.
;
; Interface:
;	chat4__record(): Get the data record for the current or a specific chat.
;	chat4__messageIndex() and chat4__messageCount(): Position in and count of messages.
;	chat4__messageText([index]): Text of current message or message with given 1-based index.
;	chat4__setPos(index):  Set the "current" message.
;	chat4__move(delta):  Move delta (plus or minus) messages from current position.
; Attempting to move outside the valid message range causes a beep and stays at the border encountered..
;
; Author:  Doug Lee


globals
; Records for chat windows, by top-level window handle. Fields:
;	hTop: The top-level window (same as the key for this record).
;	hwnd, oDoc: The actual chat window and its RichEdit document object.
;	type: The type of the window: c=channel, p=private.
;	reversed: Present and True if this is a reversed window,
;		meaning the latest chat message is at the top.
;	unit: The TOM_* text unit for moving among messages.
;	firstStart, firstIdx: The starting position and index of the first "real" message.
;	curStart: The starting position of the range for the current text unit.
;	range: A Range object set by chat4___messageIndex() and reused by chat4__messageText() to get message text.
;	size: The number of bytes in the message text (see chat4___size()).
;	end: The last recorded end of the full text range (see chat4___end()).
;	readEnd: The last read end of the full text range (see chat4___end()).
;		The above three are used to manage speaking of new text on arrival.
; Note: The caller of this module manages a reviewMessageNumber field as well.
; TODO: That should be handled in here.
	collection chat4___chats

collection function chat4__record(/*?*/ handle hwnd)
; Get or make the record for the current or given chat.
; Allows fields to be filled in that were missed at first.
if !hwnd || hwnd == 0
	hwnd = getFocus()
endIf
var handle hTop = ttutil__mainWindow(hwnd, True)
if !hTop || !isWindowVisible(hTop)
	return
endIf
var collection rec = chat4__winrecs__record(chat4___chats, hTop)
if !rec
	return
endIf

; Build or finish the record.

; Top-level window and navigation unit.
if !collectionItemExists(rec, "hTop")
	rec.hTop = hTop
	; Public messages appear to be considered paragraphs.
	; In a private message, the time/author header is a bolded paragraph,
	; and the message text is one or more normal paragraphs.
	rec.unit = 4  ; TOMParagraph
endIf

; Chat window and chat type.
if !collectionItemExists(rec, "hwnd")
	hwnd = findWindow(hTop, "RichEdit", "")
	if !hwnd
		chat4__winrecs__remove(chat4___chats, hTop)
		return
	endIf
	rec.hwnd = hwnd
	if findWindow(hTop, "SysTreeView32", "")
		rec.type = "c"
	elif findWindow(hTop, "Button", chat4___scSend)
		rec.type = "p"
		rec.reversed = True
	else
		chat4__winrecs__remove(chat4___chats, hTop)
		return
	endIf
endIf

; Chat window's RichEdit interface.
var object o
if !collectionItemExists(rec, "oDoc")
	o = getRichEditDocument(hwnd)
	if !o
		chat4__winrecs__remove(chat4___chats, hTop)
		return
	endIf
	rec.oDoc = o
else
	o = rec.oDoc
endIf

; Size and last range end for chat.
if !collectionItemExists(rec, "size")
	rec.size = chat4___size(hwnd)
endIf
if !collectionItemExists(rec, "end")
	rec.end = chat4___end(rec)
endIf
if !collectionItemExists(rec, "readEnd")
	rec.readEnd = rec.end
endIf

if !collectionItemExists(rec, "firstStart")
	rec.firststart = 0
	rec.curstart = 0
endIf

; Set rec.firststart if necessary and possible.
if !rec.firststart
	if chat4___size(rec.hwnd) < 10
		return
	endIf
	var object oRange = o.Range(0, 0)
	var int safety = 0
	while safety < 10 && oRange.char <= 13
		safety = safety +1
		oRange.start = oRange.start +1
	endWhile
	if safety >= 10
		rec.firststart = 0
		rec.firstidx = 0
		;sayInteger(chat4___size(rec.hwnd))
	else
		rec.firststart = oRange.Start
		rec.firstidx = oRange.getIndex(rec.unit)
	endIf
endIf
return rec
endFunction

int function chat4__clean(/*?*/ int clearAll)
; Clean out invalid or all chat windows.
; Returns the number of windows removed
var int nremoved = chat4__winrecs__clean(chat4___chats, clearAll)
return nremoved
endFunction

int function chat4___end(collection rec)
; Return the current endpoint of the indicated chat's text range.
; This is used in speaking new text on arrival.
var object o = rec.oDoc
if !o
	return 0
endIf
var object oRange = o.range(0, 0x7FFFFFFFL)
return oRange.end +0
endFunction

int function chat4___size(handle hwnd)
; Return the "size" of the material in the given message area window.
; This is used to detect changes.
return sendMessage(hwnd, 0x0E, 0, 0)  ; WM_GetTextLength
endFunction

int function chat4___messageIndex(int start, /*?*/ variant which)
; Return the 1-based index of the message whose start position is given, or 0 on failure..
; A start position of -1 means the last message.
; Blank lines at the top and bottom of the buffer are avoided.
; Which can bbe a window handle, a record collection, or omitted.
; This function is also used internally to set the rec.range object.
var collection rec
if !which
	; Record for currently focused chat.
	rec = chat4__record()
elif intToString(which)
	; Record for given window handle.
	rec = chat4__record(which)
else
	; Record was passed directly.
	rec = which
endIf
if !rec
	return 0
endIf

var object oDoc = rec.oDoc
var int reversed = rec.reversed
var object oRange
if start == -1
	; Find the end of the buffer.
	oRange = oDoc.Range(0, 0)
	oRange.Expand(6)  ; 6 is TOMStory
	; A blank line tends to remain at the end of the buffer.
	; -2 here avoids it.
	start = oRange.End -2
endIf
; This avoids the blank line before the first message.
if start < rec.firststart
	start = rec.firststart
endIf
oRange = oDoc.Range(start, start)
oRange.Expand(rec.unit)
rec.range = oRange
return oRange.getIndex(rec.unit) -rec.firstidx +1
endFunction

int function chat4__messageIndex(/*?*/ handle hwnd)
; Returns the 1-based index of the last-read message.
var collection rec = chat4__record(hwnd)
return chat4___messageIndex(rec.curstart, rec)
endFunction

int function chat4__messageCount(/*?*/ handle hwnd)
; Returns the number of messages in the buffer.
; A return of -1 indicates an error.
; TODO: Error checking is not here yet.
var collection rec = chat4__record(hwnd)
return chat4___messageIndex(-1, rec)
endFunction

string function chat4__messageText(/*?*/ int idx, handle hwnd)
; Return the text of the current or indicated message.
; Negative numbers index from the end.
; The current message position is not changed.
var collection rec = chat4__record(hwnd)
var string txt
if idx
	if idx < 0
		chat4___messageIndex(-1, rec)
		; sets rec.range to the last message.
		if idx != -1
			chat4___move(idx+1, rec)
		endIf
	else  ; positive idx.
		chat4___messageIndex(rec.firststart, rec)
		; sets rec.range to the first message.
		if idx != 1
			chat4___move(idx-1, rec)
		endIf
	endIf
else
	; Current message wanted.
	chat4___messageIndex(rec.curstart, rec)
	; rec.range set and we don't have to move anywhere from here.
endIf
txt = rec.range.Text
return stringTrimLeadingBlanks(stringTrimTrailingBlanks(txt))
endFunction

int function chat4___move(int delta, /*?*/ collection rec)
; Move delta messages from the current position of rec.range.
; rec.range must be set first.
; Note that this is from rec.range, not rec.curstart.
; Returns the distance moved.
if !delta
	return 0
endIf
if !rec
	rec = chat4__record()
	if !rec
		return 0
	endIf
endIf
var int curidx = chat4___messageIndex(rec.range.Start, rec)
rec.range.Collapse(-1)  ; -1 is TOMTrue, for collapse to start.
if chat4___trimDelta(delta, curidx, rec)
	beep()
endIf
rec.range.Move(rec.unit, delta)
rec.range.Expand(rec.unit)
var int result =
util__abs(curidx -chat4___messageIndex(rec.range.Start))
return result
endFunction

int function chat4___trimDelta(int byRef delta, int curidx, collection rec)
; Forces delta not to move outside the valid messages.
; Returns True if delta is changed and False if not.
if delta < 0
	if curidx+delta < 1
		delta = 1-curidx
		return True
	endIf
else
	var object oRange = rec.range.duplicate()
	var int lastidx = chat4___messageIndex(-1, rec)
	rec.range = oRange
	if curidx +delta > lastidx
		delta = lastidx -curidx
		return True
	endIf
endIf
return False
endFunction

int function chat4__move(int delta, /*?*/ handle hwnd)
; Move delta messages from the current position.
; Returns the number (absolute value) of messages moved.
var int result
var collection rec = chat4__record(hwnd)
chat4___messageIndex(rec.curstart, rec)
; That sets rec.range to reflect the current position.
result = chat4___move(delta, rec)
rec.curstart = rec.range.Start
return result
endFunction

int function chat4__setPos(int idx, /*?*/ handle hwnd)
; Set the current message to the given index.
; Negative indices count from the end.
var collection rec = chat4__record(hwnd)
chat4__messageText(idx, rec.hTop)
rec.curstart = rec.range.Start
endFunction

globals handle chat4___giSNTHwnd, int chat4___giSNTStyle
void function chat4__speakNewText(handle hwnd, int style, /*?*/ int ntenths)
; Speak text that came into the given window since it was last examined/read.
; Pass the handle to speak from, or 0 for all active chats.
; If ntenths is not 0, This function schedules itself to run that much later.
; This can be used to avoid double speech on repeated event firings.
; hwnd: The handle of the chat message window (not the top-level window for it).
; style: 0 for say nothing, 1 for current window only, 2 for current window plus alerts for other ones, and 3 for speak all messages.
; ntenths: Tenths of a second to wait before speaking, or 0.
if ntenths
	chat4___giSNTHwnd = hwnd
	chat4___giSNTStyle = style
	scheduleFunction("chat4__speakNewText", ntenths)
	return
else
	; Called from a schedule.
	hwnd = chat4___giSNTHwnd
	style = chat4___giSNTStyle
	var handle null
	chat4___giSNTHwnd = null
	chat4___giSNTStyle = 0
endIf

if hwnd
	chat4___speakNewText(hwnd, style)
	return
endIf

var string hwnds = chat4__winrecs__list(chat4___chats)
if !hwnds
	return
endIf
var int n = stringSegmentCount(hwnds, "|")
var int i = 1
while i <= n
	hwnd = stringToHandle(stringSegment(hwnds, "|", i))
	chat4___speakNewText(hwnd, style)
	i = i +1
endWhile
endFunction

void function chat4___speakNewText(handle hwnd, int style)
; Does the work for chat4__speakNewText() for one chat window.
var collection rec = chat4__record(hwnd)
if !rec
	return
endIf
var int oldSize = rec.size
var int newSize = chat4___size(hwnd)
var object o = rec.oDoc
if !o
	return
endIf
var int oldEnd = rec.end
var int oldReadEnd = rec.readEnd
var int newEnd = chat4___end(rec)
var int textSize = max(0, newEnd -oldReadEnd)
var object oRange
if rec.reversed
	oRange = o.range(0, textSize)
else
	oRange = o.range(oldReadEnd, newEnd)
endIf
if !oRange
	return
endIf
var string txt = oRange.text
rec.size = newSize
rec.End = newEnd
var int inFocus = (getTopLevelWindow(hwnd) == getTopLevelWindow(getFocus()))
if !(style == 2 && !inFocus)
	; Only holds over out-of-focus messages when style says say only alerts for them.
	rec.ReadEnd = newEnd
endIf

; Get the chat name in case we need it.
var string title
if rec.type == "c"
	; Could use the window title...
	;title = getWindowName(ttutil__mainWindow(hwnd, True))
	;title = stringSegment(title, "-", 2)
	; but that is often a stale channel name and not a server name.
	; Use the top-level room tree node's name (possibly shortened).
	title = roomtree__serverName(hwnd)
else
	; Private chats are named in the title bar.
	title = getWindowName(getTopLevelWindow(hwnd))
	; ... after the word "Messages."
	title = stringSegment(title, "-", 2)
endIf
title = stringTrimLeadingBlanks(stringTrimTrailingBlanks(title))

; Figure out what to say and say it.
if stringIsBlank(txt)
	return
endIf
if !style
	return
elif style == 3 || inFocus
	; If in focus just read the message already in txt.
	if !inFocus
		txt = formatString(chat4___msgNewMessageIn1, title, txt)
	endIf
elif style == 2 && !inFocus
	; Alert only.
	txt = formatString(chat4___msgNewMessageIn, title)
else
	; Non-focused full message.
	txt = formatString(chat4___msgNewMessageIn1, title, txt)
endIf
if txt
	sayMessage(OT_User_Requested_Information, txt)
endIf
endFunction

int function chat4__focusNext(int delta)
; Focus the next or previous chat.
; Delta should be 1 for next and -1 for previous.
chat4__registerAllChatWindows()
var string hwnds = chat4__winrecs__list(chat4___chats, "n")
if !hwnds
	return False
endIf
var handle hTop = getTopLevelWindow(getFocus())
if !hTop
	return False
elif getParent(hTop)
	; For when a dialog is in focus over a TeamTalk main window.
	hTop = getParent(hTop)
endIf
var int idx = stringSegmentIndex(hwnds, "|", intToString(hTop), True)
var int n = stringSegmentCount(hwnds, "|")
if !idx
	if delta > 0
		idx = n
	else
		idx = 1
	endIf
endIf
idx = idx +delta
if idx < 1
	idx = idx +n
elif idx > n
	idx = idx -n
endIf
var handle hwnd = stringToHandle(stringSegment(hwnds, "|", idx))
if hwnd == hTop
	sayMessage(OT_Error, chat4___msgNoMoreWindows)
	return True
endIf
if isWindowDisabled(hwnd)
	hwnd = chat4___getActiveDialog(hwnd)
	if !hwnd
		return False
	endIf
endIf
setFocus(hwnd)
var int safety = 50
while safety
	if getTopLevelWindow(getFocus()) == getTopLevelWindow(hwnd)
		safety = 1
	elif !getFocus() && getForegroundWindow() == getTopLevelWindow(hwnd)
	&& MSAA__windowState(getForegroundWindow(), 0, ObjID_Window) & State_System_Invisible
		typeKey("Enter")
		safety = 1
	else
		pause()
	endIf
	safety = safety -1
endWhile
return True
endFunction

handle function chat4___getActiveDialog(handle hwnd0)
; Get the dialog that is disabling the given window.
hwnd0 = getTopLevelWindow(hwnd0)
var handle hwnd = getFirstWindow(hwnd0)
while hwnd
	if getParent(hwnd) == hwnd0
		return hwnd
	endIf
	hwnd = getNextWindow(hwnd)
endWhile
return 0
endFunction

void function chat4__registerAllChatWindows()
; Build or update the list of all valid chat windows.
var string thisOwner = getWindowOwner(getTopLevelWindow(getFocus()))
var handle hwnd = findTopLevelWindow("#32770", "")
while hwnd
	if isWindowVisible(hwnd)
	&& stringCompare(getWindowOwner(hwnd), thisOwner, True) == 0
	&& !getParent(hwnd)
		chat4__record(hwnd)
	endIf
	hwnd = getNextWindow(hwnd)
endWhile
chat4__clean()
endFunction


;-------------------------------- Module winrecs -------------------------------
; A manager for data structures (records) indexed by a window handle.
;
; Author:  Doug Lee of SSB BART Group

collection function chat4__winrecs__record(collection byRef recs, handle hwnd, optional int noCreate)
; Return the record for the given window.
; Registers this window in the set if necessary.
; Also removes stale records for windows that no longer exist.
; The caller must handle record structure creation.
; Suggestion: Use an "hwnd" property so there's a uniform way to test for a new record.
; Pass noCreate if you need to avoid creating a new record.
chat4__winrecs__clean(recs)
if !hwnd then
	return
endIf
var string shwnd let shwnd = intToString(hwnd)
var collection rec
if !recs then
	let recs = new collection
endIf
if !collectionItemExists(recs, shwnd) then
	if noCreate
		return
	endIf
	; We have to make a new record for this one.
	let rec = new collection
	recs[shwnd] = rec
	; Caller must detect that this record is empty and build it.
else
	let rec = recs[shwnd]
endIf
return rec
endFunction

int function chat4__winrecs__remove(collection recs, handle hwnd)
; Remove a window's record.
var string shwnd let shwnd = intToString(hwnd)
if !collectionItemExists(recs, shwnd) then
	return False
endIf
collectionRemoveItem(recs, shwnd)
return True
endFunction

int function chat4__winrecs__clean(collection recs, /*?*/ int clearAll)
; Remove from recs any records for windows that no longer exist.
; If clearAll is True, removes all records.
; Returns the number of records removed.
; Note that this function is called by chat4__winrecs__record(),
; so manual use of it is optional.
var string shwnd
var int nremoved = 0
if clearAll then
	var int n = 0
	forEach shwnd in recs
		n = n +1
	endForEach
	collectionRemoveAll(recs)
	return n
endIf
var handle hwnd
forEach shwnd in recs
	let hwnd = stringToHandle(shwnd)
	if !getWindowStyleBits(hwnd) then
		if chat4__winrecs__remove(recs, hwnd) then
			let nremoved = nremoved +1
		endIf
	endIf
endForEach
return nremoved
endFunction

string function chat4__winrecs__list(collection recs, /*?*/ string how)
; Return a |-delimited list of the registered window handles.
; By default, handles are returned in the order in which they were added to the set.
; If how is "n", they are returned in numeric order.
; If how is anything else, it is considered a function name or call.
; See util__compare() for details.
var string hwnds, string shwnd
forEach shwnd in recs
	let hwnds = hwnds +"|" +shwnd
endForEach
let hwnds = stringChopLeft(hwnds, 1)
if !how then
	return hwnds
endIf
; Kludge: Make XList out of hwnds.
let hwnds = "|" +hwnds +"|"
let hwnds = chat4__winrecs__xlist__sort(hwnds, how, False)
; Remove the extra delimiters added above.
let hwnds = stringChopLeft(stringChopRight(hwnds, 1), 1)
return hwnds
endFunction


;__________________________________ Module log _________________________________
; Allow logging to a text file.
;
; Interface:  chat4__winrecs__log__open, chat4__winrecs__log__isOpen, chat4__winrecs__log__close, chat4__winrecs__log__write, chat4__winrecs__log__setUserID.
; Basic usage: chat4__winrecs__log__open() from AutoStartEvent, chat4__winrecs__log__write("...") to log something.
; Usage:
;	- Call chat4__winrecs__log__open() from AutoStartEvent.
;	- Logging will occur by default only if the log file already exists (good for debugging during development).
;	- Call as chat4__winrecs__log__open("", True) if you want logging to occur even if the log file doesn't already exist.
;	- Pass a file name instead of "" for a custom file name (rarely needed).
;	- Call chat4__winrecs__log__write("Oops I saw an error") to log something.
;	- Use chat4__winrecs__log__write("Oops this error needs to speak too", True) to speak what is being logged.
;	- Call chat4__winrecs__log__close() to disable logging (rarely needed).
;
; Author:  Doug Lee of SSB BART Group

; Constants for chat4__winrecs__log__write and related functions.
; Not localized because log text is not localized.

; Used as a user ID when none is found.
const chat4__winrecs__log___msgNoName = "<no name>"
; What "name" is used for the caller string when JAWS is too old for getScriptCallStack().
; %1 is the JAWS version.
const chat4__winrecs__log___msgJAWSTooOld = "(caller unknown; JAWS version (%1) too old)"
; Name used for the caller when it can't be determined for an unknown reason,
; probably a getScriptCallStack return value format surprise.
const chat4__winrecs__log___msgNoCaller = "(caller unknown)"

globals
; FileSystemObject and log stream object obtained from it.
	object chat4__winrecs__log___fso,
	object chat4__winrecs__log___logstream,
; Name of file.
	string chat4__winrecs__log___fileName,
	int chat4__winrecs__log___canCreate,
	string chat4__winrecs__log___userID,
	string chat4__winrecs__log___machineID

int function chat4__winrecs__log___setup()
; Internal function called to set things up when necessary before any log text is generated.
if !stringIsBlank(chat4__winrecs__log___userID) then
	return True
endIf
let chat4__winrecs__log___fso = createObjectEx("Scripting.FileSystemObject", False)
if !chat4__winrecs__log___fso then
	return False
endIf
let chat4__winrecs__log___machineID = ""
var object oNet let oNet = createObjectEx("WScript.Network", False)
if oNet then
	let chat4__winrecs__log___machineID = oNet.ComputerName
endIf
if stringIsBlank(chat4__winrecs__log___machineID) then
	let chat4__winrecs__log___machineID = chat4__winrecs__log___msgNoName
endIf
let chat4__winrecs__log___userID = getJAWSUserName()
if stringIsBlank(chat4__winrecs__log___userID) && oNet then
	let chat4__winrecs__log___userID = oNet.UserName
endIf
if stringIsBlank(chat4__winrecs__log___userID) then
	let chat4__winrecs__log___userID = chat4__winrecs__log___msgNoName
endIf
return True
endFunction

int function chat4__winrecs__log__setUserID(string id)
; Optional means of setting a custom user ID.
; If not called, the value of getJAWSUserName() is used.
let chat4__winrecs__log___userID = id
return True
endFunction

int function chat4__winrecs__log__close()
; Close the log file.
chat4__winrecs__log___close()
let chat4__winrecs__log___fileName = ""
return True
endFunction

int function chat4__winrecs__log___setFileName(string name)
; Set the log file name.
; The name can be passed; otherwise <basename>.log is used.
; <basename> is getActiveConfiguration() or, failing that, the basename of getAppFileName().
; The log file is located in the JAWS user folder.
; Helper for chat4__winrecs__log__open().
var string basename
let basename = name
if stringRight(basename,4) == ".log" then
	let basename = stringChopRight(basename, 4)
endIf
if !basename then
	let basename = getActiveConfiguration()
endIf
if !basename then
	let basename = getAppFileName()
	var int pos let pos = stringContainsFromRight(basename, ".")
	if pos > 0 then
		let basename = stringLeft(basename, pos-1)
	endIf
endIf
let chat4__winrecs__log___fileName = getJAWSSettingsDirectory() +"\\" +basename
if stringRight(chat4__winrecs__log___fileName, 4) != ".log" then
	let chat4__winrecs__log___fileName = chat4__winrecs__log___fileName +".log"
endIf
return True
endFunction

int function chat4__winrecs__log__open(/*?*/ string name, int canCreate)
; Open and optionally create the log file.
; The name can be passed; otherwise <basename>.log is used.
; <basename> is getActiveConfiguration() or, failing that, the basename of getAppFileName().
; The log file is located in the JAWS user folder.
; Call from autoStartEvent, or logging will not occur.
; Returns True if the file is opened and False if not.
if !chat4__winrecs__log___setup() then
	return False
endIf
let chat4__winrecs__log___canCreate = canCreate
chat4__winrecs__log__close()
if !chat4__winrecs__log___setFileName(name) || !chat4__winrecs__log___open() then
	let chat4__winrecs__log___fileName = ""
	return False
endIf
return True
endFunction

int function chat4__winrecs__log___open()
; Raw open/create.  Sets chat4__winrecs__log___logstream.
; Returns True on success and False on failure.
if chat4__winrecs__log___logstream then
	return True
endIf
if fileExists(chat4__winrecs__log___fileName) then
	let chat4__winrecs__log___logstream = chat4__winrecs__log___fso.openTextFile(chat4__winrecs__log___fileName, 8)  ; Append
elif chat4__winrecs__log___canCreate then
	let chat4__winrecs__log___logstream = chat4__winrecs__log___fso.CreateTextFile(chat4__winrecs__log___fileName)
endIf
if !chat4__winrecs__log___logstream then
	return False
endIf
scheduleFunction("chat4__winrecs__log___close", 30)
return True
endFunction

int function chat4__winrecs__log___close()
; Raw cloase; unsets chat4__winrecs__log___logstream.
var variant null
if !chat4__winrecs__log___logstream then
	return True
endIf
chat4__winrecs__log___logstream.Close()
let chat4__winrecs__log___logstream = null
return True
endFunction

int function chat4__winrecs__log___isOpen()
; Returns True if the log file is actually open and False if not.
if chat4__winrecs__log___logstream then
	return True
endIf
return False
endFunction

int function chat4__winrecs__log__isOpen()
; Returns True if the log file is open and False if not.
; "Open" here means can be written to but may not necessarily mean the file is physically opened at the moment.
if chat4__winrecs__log___fileName then
	return True
endIf
return False
endFunction

string function chat4__winrecs__log___getStamp()
; Returns a formatted date/time stamp for the log file.
var string tm let tm = sysGetTime("HH:mm:ss")
var string dt let dt = sysGetDate("ddd MMM dd yyyy")
return formatString("%1 %2", tm, dt)
endFunction

int function chat4__winrecs__log__write(string s, /*?*/ int alsoSpeak)
; Write a string to the log file, and also optionally speak it.
if alsoSpeak then
	sayMessage(OT_Error, s)
endIf
if !chat4__winrecs__log__isOpen() then
	return False
endIf
if !chat4__winrecs__log___open() then
	return False
endIf
var string buf let buf = formatString(
	"*** %1 %2@%3 (%4)\n%5:\n",
	chat4__winrecs__log___getStamp(), chat4__winrecs__log___userID, chat4__winrecs__log___machineID, getAppFileName(),
	chat4__winrecs__log___getCaller()
) +s +"\n\n"
; Standardize any CR/LF characters in buf before writing.
let buf = stringReplaceChars(buf, "\r\n", "\n")
let buf = stringReplaceSubstrings(buf, "\n", "\r\n")
chat4__winrecs__log___logstream.write(buf)
return True
endFunction

string function chat4__winrecs__log___getCaller()
; Get and return the caller of this logging system.
; Helper for chat4__winrecs__log__write().
if getJFWVersion() < 1000000 then
	var string jver let jver = stringSegment(util__getJAWSInfo(), "|", 1)
	return formatString(chat4__winrecs__log___msgJAWSTooOld, jver)
endIf
var string stack let stack = util__callStackInfo(2, "s", 2)
if !stringIsBlank(stack) then
	return stack
endIf
return formatString("%1\nStack begin:\n%2\nStack end.",
	chat4__winrecs__log___msgNoCaller,
	util__vcast(getScriptCallStack())
)
endFunction

string function chat4__winrecs__log___splitOff(string byRef s, string sep, /*?*/ int fromRight)
; Split what's before sep off and return it.
; If fromRight is True, split everything before the last sep off and return that.
; Returns by reference s with both that and sep removed.
var string s1
var int pos
if fromRight then
	let pos = stringContainsFromRight(s, sep)
else
	let pos = stringContains(s, sep)
endIf
if pos then
	let s1 = stringLeft(s, pos-1)
	let s = stringChopLeft(s, pos+stringLength(sep)-1)
else
	let s1 = s
	let s = ""
endIf
return s1
endFunction

;________________________________ End module log _______________________________


;_________________________________ Module xlist ________________________________
; XList, the arbitrary list (array and hash emulation) module.
;
; Terminology:
;	XList - An arbitrary list of items (an array, basically).
;	XLHash - An XList in which odd-numbered items are keys and even-numbered items are their values.
;
; Implementation Notes:
; XLists always begin and end with a delimiter; a null list is just a single delimiter character.
; This means it is NOT possible to use boolean (if sl) or length (if stringLength(sl)) to check for emptiness; use isEmpty or itemCount.
; Objects can not be stored in an XList directly, and attempting this will cause an error message to speak.
; Translation Notes:  The only translatable strings appear in SayMessage calls.
;
; Author:  Doug Lee
;
; Revision History:  See version control logs.

const
; Indexes used to request the character tables supported by this module.
	chat4__winrecs__xlist___CharANSI = 1,
; The corresponding character tables.
; The double quote (") and backslash characters must be escaped.
	chat4__winrecs__xlist___CharANSICharList = "\1\2\3\4\5\6\7\8\9\10\11\12\13\14\15\16\17\18\19\20\21\22\23\24\25\26\27\28\29\30\31 !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\127\128\129\130\131\132\133\134\135\136\137\138\139\140\141\142\143\144\145\146\147\148\149\150\151\152\153\154\155\156\157\158\159\160\161\162\163\164\165\166\167\168\169\170\171\172\173\174\175\176\177\178\179\180\181\182\183\184\185\186\187\188\189\190\191\192\193\194\195\196\197\198\199\200\201\202\203\204\205\206\207\208\209\210\211\212\213\214\215\216\217\218\219\220\221\222\223\224\225\226\227\228\229\230\231\232\233\234\235\236\237\238\239\240\241\242\243\244\245\246\247\248\249\250\251\252\253\254\255"

string function chat4__winrecs__xlist__sort(string xl, string how, int sortDescending)
; Sorts xl and returns the result.
; How is as for util__compare().
; This is a stable sort; equal items are not swapped.
; TODO:  The algorithm is slow:  sorting 200 items took nine seconds on a Sony Vaio 1.2 GHz Centrino machine (Windows XP, JAWS 7.10, DGL, 2006-08-25).
var
	string xl1,
	string tmp, string firstVal, int firstIdx,
	int i, int n, int cmp
; Using the below trick instead of chat4__winrecs__xlist__new() preserves the separator from xl on return.
let xl1 = stringLeft(xl, 1)
let n = chat4__winrecs__xlist__itemCount(xl)
if !n then
	return xl1
elif n == 1 then
	; The commented-out approach guarantees an XList structure return;
	; the faster plan, in effect, allows single-item strings to pass through unchecked.
	;chat4__winrecs__xlist__addItem(xl1, chat4__winrecs__xlist__getItem(xl, 1))
	;return xl1
	return xl
endIf

; We have two or more items.
while n > 1
	let firstVal = chat4__winrecs__xlist__getItem(xl, 1)
	let firstIdx = 1
	let i = 2
	while i <= n
		let tmp = chat4__winrecs__xlist__getItem(xl, i)
		let cmp = util__compare(firstVal, tmp, how)
		if sortDescending then
			let cmp = cmp * (-1)
		endIf
		if cmp > 0 then
			let firstVal = string__ref(tmp)
			let firstIdx = i
		endIf
		let i = i +1
	endWhile
	; firstVal and firstIdx are the value and index of the "first" value, respectively.
	; "First" means smallest for ascending and greatest for descending sorts.
	chat4__winrecs__xlist__addItem(xl1, firstVal)
	chat4__winrecs__xlist__delete(xl, firstIdx, 1)
	let n = n -1
endWhile
; What's left is the max item.
chat4__winrecs__xlist__addItem(xl1, chat4__winrecs__xlist__getItem(xl, 1))
return xl1
endFunction

int function chat4__winrecs__xlist__delete(string byRef lst, int i, int n)
; Deletes n elements starting at element i.
var
	int pos1, int pos2
let pos1 = chat4__winrecs__xlist___indexPos(lst, i)
if !pos1 || pos1 == stringLength(lst)+1 then
	return False
endIf
let pos2 = chat4__winrecs__xlist___indexPos(lst, i+n)
if !pos2 || pos2 == stringLength(lst)+1 then
	let lst = stringLeft(lst, pos1-1)
	return True
endIf
let lst = string__cat(stringLeft(lst, pos1-1), stringChopLeft(lst, pos2-1))
return True
endFunction

int function chat4__winrecs__xlist__addItem(string byRef lst, variant item)
; Append item to lst.
; If lst is null, it will be created first; but if it is non-null and not an XList, a mess is likely.
; Returns True on success and False on failure.
; On failure, lst is not changed (except for maybe being initialized if null).
if !stringLength(item) && item then
	sayMessage(OT_No_Disable, "Can't store an object in an XList")
	return False
endIf
if !chat4__winrecs__xlist___settleDelim(lst, item, 1) then
	return False
endIf
let lst = string__cat(lst, string__cat(item, stringLeft(lst, 1)))
return True
endFunction

variant function chat4__winrecs__xlist__getItem(string lst, int i)
; Returns the ith element of lst.
; If i is greater than the item count, the null string is returned.
var string val
if i < 1 || !stringLength(lst) then
	return ""
endIf
let val = stringSegment(lst, stringLeft(lst, 1), i+1)
return val
endFunction

int function chat4__winrecs__xlist__itemCount(string lst)
; Returns the number of items in lst.
; If lst is not a proper XList, returns 0.
if !stringLength(lst) then
	return 0
endIf
return string__segmentCount(lst, stringLeft(lst, 1)) -2
endFunction

int function chat4__winrecs__xlist___settleDelim(string byRef lst, string item, int dlen)
; Internal function to manage the changing of delimiters as needed.
; Must be called whenever material is being added to an XList.
; dlen is the length of lst's current delimiter (1 except when sent via fromJAWSList).
; Returns True if a valid delimiter is in place on exit and False if there is no safe delimiter available.
; A "valid" delimiter is a one-character delimiter not found in lst (except as the current delimiter) or item.
var
	string delim,
	int c0, int c, string ch,
	int done
if !stringLength(lst) then
	let lst = chat4__winrecs__xlist__new()
endIf
let delim = stringLeft(lst, dlen)
if stringContains(item, delim) || dlen != 1 then
	let c = chat4__winrecs__xlist__charToANSI(stringLeft(lst, 1))
	let c0 = c
	let done = False
	while !done
		let c = c +1
		if c > 254 then
			; 255 may not write correctly to a file, and 0 is not possible in JAWS.
			let c = 1
		elif c >= 8 && c <= 13 then
			; backspace, tab, LF, RF(?), FF, CR:
			; These characters are unfriendly when written to a text file, so they are avoided.
			let c = 14
		elif c == 26 || c == 27 then
			; DOS/Windows EOF, ESC
			let c = 28
		endIf
		let ch = chat4__winrecs__xlist__charFromANSI(c)
		if c == c0 then
			; We've come all the way around; there is no safe delimiter for this situation.
			beep()
			sayMessage(OT_No_Disable, "xlist:  Out of delimiters")
			return False
		elif !stringContains(lst, ch) && !stringContains(item, ch) then
			; We have a winner!
			chat4__winrecs__xlist___setDelim(lst, ch, dlen)
			let delim = ch
			let done = True
		endIf
	endWhile
endIf
return True
endFunction

int function chat4__winrecs__xlist__charToANSI(string c)
; Returns the ANSI code for the first character in c, or 0 if c is null.
return chat4__winrecs__xlist___getCharCode(c, chat4__winrecs__xlist___CharANSI)
endFunction

string function chat4__winrecs__xlist__charFromANSI(int i)
; Returns the character corresponding to the given ANSI code.
return chat4__winrecs__xlist___getChar(i, chat4__winrecs__xlist___CharANSI)
endFunction

string function chat4__winrecs__xlist__new()
; Create and return a new XList.
return "\1"
endFunction

int function chat4__winrecs__xlist___indexPos(string lst, int i)
; Returns the character position in lst of the first character of the ith element.
; Returns 0 if there is no such element.
; Exception:  If i == itemCount+1, returns stringLength(lst)+1.
var
	string delim,
	int len,
	int pos,
	int result
if i < 1 then
	return 0
endIf
let delim = stringLeft(lst, 1)
let len = stringLength(lst)
if !len then
	return 0
elif len == 1 then
	return 1
endIf
let result = 0
while i
	let pos = stringContains(lst, delim)
	if !pos then
		return 0
	endIf
	let result = result +pos
	let lst = stringChopLeft(lst, pos)
	let i = i -1
endWhile
return result +1
endFunction

int function chat4__winrecs__xlist___getCharCode(string s, int whichList)
; Converts the first character in s to the corresponding code based on the indicated character table.
; If s is null, 0 is returned.
; Can't handle \000 because this seems to end JAWS strings.
; In fact, including \000 anywhere in a string causes it to appear empty!
; [DGL, 02/28/03]
let s = stringLeft(s, 1)  ; in case s contains more than one character
if !stringLength(s) then
	return 0
endIf
return stringContains(chat4__winrecs__xlist___getCharList(whichList), s)
endFunction

string function chat4__winrecs__xlist___getChar(int i, int whichList)
; Converts the given code to the corresponding character from the indicated character table.
; If i is less than 1 or above the length of the table, the null string is returned.
var string charList
if i <= 0 then
	return ""
endIf
let charList = chat4__winrecs__xlist___getCharList(whichList)
if i > stringLength(charList) then
	return ""
endIf
return substring(charList, i, 1)
endFunction

string function chat4__winrecs__xlist___getCharList(int whichList)
; Returns the requested character list.  This is to avoid duplicating huge constants in the script binary.
if whichList == chat4__winrecs__xlist___CharANSI then
	return chat4__winrecs__xlist___CharANSICharList
endIf
return ""
endFunction

int function chat4__winrecs__xlist___setDelim(string byRef lst, string ch, int dlen)
; Sets lst's delimiters to ch.
; dlen is the length of lst's current delimiter (1 except when sent via fromJAWSList).
var
	string delim
let delim = stringLeft(lst, dlen)
if stringLength(ch) == 1 && dlen == 1 then
	let lst = string__ref(stringReplaceChars(lst, delim, ch))
else
	; {To/From}JAWSList() can send a multi-character delimiter here, but no proper XList will use one.
	let lst = string__ref(stringReplaceSubstrings(lst, delim, ch))
endIf
return True
endFunction

;_______________________________ End module xlist ______________________________

;------------------------------ End module winrecs -----------------------------

;=============================== End module chat4 ==============================


;=============================== Module roomtree ===============================
; Room tree interface for TeamTalk Classic.
;
; Author:  Doug Lee

/*
Design info:
The following apply to the SysTreeView32 control supported by this module,
and unless otherwise noted, this module depends on these features to
work properly:

Each node has a child ID; there are no full object descendants (this
is normal for a treeview).

Child IDs are numerically contiguous at first, though they may not
start at 1.  (Most treeviews do not allow any assumptions about childID
contiguousness.)  As people join and leave the server though, holes in
the childID contiguousness develop.

A childID is never reused, even when a user leaves one server and
joins another, which means childIDs increase monotonically for as long
as the user keeps TeamTalk open.  (This fact is not used here.)

Traversing by incrementing childID seems to go in breadth-first order
(not used here).

Traversing by AccNavigate(NavDir_Down,...) goes in depth-first order
(normal for treeviews).

Spatial navigation provides correct parent/child relationships (normal
for treeviews).

The tree consists of all nodes (channels and users) immediately on server
join.  (Normal trees can leave out subnodes until the user expands them.)

Channel nodes are denoted by a parenthesized number at the end of
accName, (0) for a channel with no members.

A channel can have both member and channel descendants, but the number
in its name represents only members in that channel.

A channel's Open state flag is not reset when the channel empties even
though there may no longer be descendants.
(This module makes no use of state flags; this is just written here
because it is unusual.)

This information comes from research by DGL on 2010-05-30,
supplemented on 2010-08-30.
*/

globals
; A list of tree window handles.
	string roomtree___hwnds,
; A collection of storage structures by server tree window handle.
; Fields:
;	hwnd, client:  The window handle and MSAA client object for the tree.
;	nodes:  The raw list of node names at last check.
	collection roomtree___trees,
; First and last childIDs in numerical order.
; Used for byAge navigation.
	int roomtree___firstID, int roomtree___lastID,
; Used to avoid very time-consuming getColorAtPoint calls on Win8:
; 0 means untested, 1 means ok, -1 means too slow to use.
	int roomtree___pixelTestCondition

int function roomtree__reportMemberships(int includeList, /*?*/ int startNode)
; Report who is where in the current TeamTalk server's channel tree.
; includeList: True to include names, False to show only counts per channel.
; TODO: This function is not fully written.
var collection tree = roomtree___treeRecord()
if !tree
	sayMessage(OT_Error, roomtree___msgNoTree)
	return False
endIf
var object o = tree.client
if !startNode
	; Start on the root node.
	startNode = 1
endIf
var string names = ""
var int childID = o.accNavigate(7, startNode)  ; 7 is NavDir_FirstChild
var int safety = 1000
var string name, int memberCount
while childID && safety
	roomtree___getNameAndCount(o.accName(childID), name, memberCount)
	if includeList
		if memberCount == -1
			; A member, not a channel.
		endIf
	else
	endIf
	childID = o.accNavigate(2, childID)  ; 2 is NavDir_Down
	safety = safety -1
endWhile
; Safety ran out; a bad thing.
beep()
return False
endFunction

collection function roomtree___treeRecord(/*?*/ handle hwnd)
; Returns the record for the given or currently focused TeamTalk's channel tree.
; Registers this tree in roomtree___trees if necessary.
; Also removes stale records for windows that no longer exist.
; Does not update the record itself from the tree.
roomtree___cleanOutStaleWindows()
var variant null
if hwnd
	hwnd = findWindow(hwnd, "SysTreeView32", "")
	if !hwnd
		return
	endIf
else
	hwnd = roomtree__currentTreeWindow()
	if !hwnd
		return
	endIf
endIf
var string shwnd = intToString(hwnd)
var collection tree, object o, int childID
if !collectionItemExists(roomtree___trees, shwnd)
	; We have to make a new record for this one.
	o = getObjectFromEvent(hwnd, -4, 0, childID)  ; -4 is ObjID_Client
	if !o
		return False
	endIf
	roomtree___hwnds = formatString("%1|%2", roomtree___hwnds, shwnd)
	tree = new collection
	if !roomtree___trees
		roomtree___trees = new collection
	endIf
	roomtree___trees[shwnd] = tree
	tree.hwnd = hwnd
	tree.client = o
	var collection nodes nodes = new collection
	tree.nodes = nodes
else
	tree = roomtree___trees[shwnd]
endIf
return tree
endFunction

int function roomtree___cleanOutStaleWindows()
; Remove from roomtree___trees any records for windows that no longer exist.
; Returns the number of records removed.
var int n = stringSegmentCount(roomtree___hwnds, "|")
var int nremoved = 0
var string shwnd
while n
	shwnd = stringSegment(roomtree___hwnds, "|", n)
	if getWindowClass(stringToHandle(shwnd)) != "SysTreeView32"
		nremoved = nremoved +1
		collectionRemoveItem(roomtree___trees, shwnd)
		if !string__segmentRemove(roomtree___hwnds, "|", shwnd)
			chat4__winrecs__log__write("Failed to remove handle " +shwnd +" from the tree list.")
		endIf
	endIf
	n = n -1
endWhile
return nremoved
endFunction

handle function roomtree__currentTreeWindow()
; Return the handle of the channel tree for the focused TeamTalk instance,
; or return null on error.
var handle hwnd = ttutil__mainWindow()
if !hwnd
	return 0
endIf
hwnd = findWindow(hwnd, "SysTreeView32", "")
if !hwnd
	return 0
endIf
return hwnd
endFunction

int function roomtree___getNameAndCount(string s, string byRef name, int byRef count)
; Return the name and count from the given channel tree node name text.
; If the text represents a user name and not a channel (no count included),
; the returned count will be -1.
s = stringTrimLeadingBlanks(stringTrimTrailingBlanks(s))
var int pos = stringContainsFromRight(s, " ")
if !pos || substring(s, pos+1, 1) != "("
|| stringRight(s, 1) != ")"
	count = -1
	name = s
	return True
endIf
name = stringLeft(s, pos-1)
var string tmp = stringChopRight(stringChopLeft(s, pos+1), 1)
var int n = stringLength(tmp)
var int i = 1
while i <= n
	if !stringContains("0123456789", substring(tmp, i, 1))
		name = s
		count = -1
		return True
	endIf
	i = i +1
endWhile
count = stringToInt(tmp)
return True
endFunction

int function roomtree__isMember(object o, int childID)
; Returns True if this is a member node, as opposed to a channel.
var string name, int count
if !o.accRole(childID)
	; This happens for a skipped childID in byAge navigation.
	return False
elif !roomtree___getNameAndCount(o.accName(childID), name, count)
	; Unrecognized nodes are called members here.
	beep()
	return True
endIf
return (count == -1)
endFunction

int function roomtree__getMember(int childID, string which, /*?*/ int byAge, int alsoSpeak)
; Get the child ID of a member based on which and, if passed, byAge.
; "Member" means person, as opposed to channel.
; If childID is 0 or invalid, starts as appropriate from an end of the list or the current position.
; If byAge is False or not passed, uses tree order.
; If byAge is True, attempts to order members chronologically by when they last joined a channel.
; If alsoSpeak is True, speaks error messages as appropriate.
; Values of which:
;	first, last: First or last member.
;	next, prev: Next or previous member from given position.
;	nextCyclic, prevCyclic: Same but looping around the member list.
; On success, returns the childID of the member sought.
; On error, returns 0.
var collection tree = roomtree___treeRecord()
if !tree
	if alsoSpeak
		sayMessage(OT_Error, roomtree___msgNoTree)
	endIf
	return False
endIf
var object o = tree.client
if byAge
	roomtree___setFirstAndLastChildIDs(o)
endIf

; Handle next/prev requests, or if appropriate, make first/last requests out of them.
if which == "next" || which == "prev"
	if !childID
		childID = o.accFocus
	endIf
	if childID
		childID = roomtree___navNext(o, childID, (which == "next"), byAge, True)
	endIf
	if !childID
		; No current position or no next one,
		; so these become first/last requests in cyclic mode.
		; They also do this if no node is focused yet,
		; which happens when a server connection is first opened.
		if !stringContains(stringLower(which), "cyclic") && o.accFocus
			if alsoSpeak
				sayMessage(OT_Error, roomtree___msgNoMoreMembers)
			endIf
			return 0
		endIf
		if which == "next"
			which = "first"
		else  ; prev
			which = "last"
		endIf
	endIf
endIf

; Handle first and last, both by tree and by age.
if which == "first"
	; The first childID is also the oldest.
	childID = o.accNavigate(NavDir_FirstChild, 0)
	childID = roomtree___navNext(o, childID, True, byAge, False)
elif which == "last"
	if byAge
		childID = roomtree___lastID
	else
		childID = o.accNavigate(NavDir_LastChild, 0)
	endIf
	childID = roomtree___navNext(o, childID, False, byAge, False)
endIf
if !childID
	if alsoSpeak
		sayMessage(OT_Error, roomtree___msgNoMembers)
	endIf
endIf
return childID
endFunction

int function roomtree___navNext(object o, int childID, int forward, int byAge, int skipThisOne)
; Return the next childID in the requested direction and sort order.
; If skipThisOne is True, the given node is not considered, otherwise it is.
; For byAge True, assumes roomtree___firstID and roomtree___lastID are set.
; Helper for roomtree__getMember().
var int dir
if byAge
	if forward
		dir = 1
	else
		dir = -1
	endIf
else
	if forward
		dir = NavDir_Next
	else
		dir = NavDir_Previous
	endIf
endIf

while childID
	if !skipThisOne && roomtree__isMember(o, childID)
		return childID
	endIf
	if byAge
		childID = childID +dir
		if childID < roomtree___firstID || childID > roomtree___lastID
			; We hit an edge of the valid childIDs.
			childID = 0
		endIf
	else
		childID = o.accNavigate(dir, childID)
	endIf
	skipThisOne = False
endWhile
return 0
endFunction

void function roomtree___setFirstAndLastChildIDs(object o)
; Sets roomtree___firstID and roomtree___lastID, used for byAge navigation.
; Helper for roomtree__getMember().
var int childID = o.accNavigate(NavDir_FirstChild, 0)
; This has to be the "oldest" childID because it represents the tree root,
; which is the first childID created.
roomtree___firstID = childID
; We know the last ID can't be less than this...
var int maxID = roomtree___firstID +o.accChildCount -1
; but without a loop, we can't know what the highest one really is.
while childID
	if childID > maxID
		maxID = childID
	endIf
	childID = o.accNavigate(NavDir_Next, childID)
endWhile
roomtree___lastID = maxID
endFunction

int function roomtree__move(string which, int byAge)
; Move to another member node.
; Returns True on success and False on failure.
var int childID = roomtree__getMember(0, which, byAge, True)
if !childID
	return False
endIf
var collection tree = roomtree___treeRecord()
if !tree
	return False
endIf
var object o = tree.client
var int focusID = o.accFocus +0
if childID == focusID
	; The indirect call avoids compilation problems before JAWS 8,
	; which are otherwise caused by the passing of a parameter.
	; The parameter avoids position info announcement like "1 of 6."
	beep()
	formatStringWithEmbeddedFunctions("<sayObjectActiveItem(0)>")
else
	; Leaving out SelFlag_TakeFocus avoids the following anomaly:
	; Focus remains, in MSAA and by window, on the right thing,
	; but Alt doesn't work, Alt+F4 destroys the tree leaving the app running,
	; and Tab and then Alt+F4 causes a busy loop terminated only by Alt+Tab.
	; Using TakeFocus+TakeSelection but calling AccSelect twice solves
	; the menu problem but causes the loop issue to occur on Alt+F4 without Tab.
	o.accSelect(SelFlag_TakeSelection, childID)
	if stringCompare(roomtree___treePath(o, childID), roomtree___treePath(o, focusID), True) != 0
		roomtree___sayNode()
	endIf
endIf
return True
endFunction

void function roomtree___sayNode()
; Say the path to the current tree node.
var collection tree = roomtree___treeRecord()
if !tree
	return
endIf
var object o = tree.client
var string path = roomtree___treePath(o, o.accFocus)
sayUsingVoice(VCTX_Message, path, OT_Message)
endFunction

string function roomtree___treePath(object o, int childID)
; Return the path from root to given tree node.
; Omits the given node and the top-level node.
var int safety = 50
var string path = ""
var string name, int cnt
while safety
	childID = o.accNavigate(NavDir_Left, childID)
	if childID && o.accNavigate(NavDir_Left, childID)
		if !roomtree___getNameAndCount(o.accName(childID), name, cnt)
			beep()
			return ""
		endIf
		if path
			path = ", " +path
		endIf
		path = name +path
	else
		safety = 1  ; exits loop
	endIf
	safety = safety -1
endWhile
return path
endFunction

string function roomtree__iconStatus()
; Indicates if the user at the active cursor position is transmitting sound and/or idle.
; Includes code to dodge this test if it takes too long,
; as it can do on Windows 8.
if roomtree___pixelTestCondition == -1
	; GetColorAtPoint is too slow to use.
	return ""
endIf
var handle hwnd, object o, int childID
hwnd = getCurrentWindow()
o = getCurrentObject(childID)
if !o
	return ""
elif !childID
	; On top level of tree, no icons here.
	return ""
	elif !o.accRole(childID)
		; Can happen when the tree is collapsed by a disconnect:
		; GetFocusObject() can return a stale childID.
		; [DGL, 2011-12-29, JAWS 13.0.704]
		return ""
endIf
roomtree___scrollTree(hwnd, o, childID, 20)
if !roomtree__isMember(o, childID)
	; We don't have icon statuses for channel nodes.
	return ""
endIf
var int left, int right, int top, int bottom
if !MSAA__rect(o, childID, left, right, top, bottom)
	return ""
endIf
var int x = left-4
var int y = (top+bottom) /2
var int n = x -getWindowLeft(getCurrentWindow())
n = max(n, 0)
n = min(n, 15)
var string flags = ""
var int clr
var int isSending = False
var int isIdle = False
var int foundIcon = False
var int ptc = getTickCount()
while n
	clr = getColorAtPoint(x, y)
	if clr == 0x0000FFL
		isIdle = True
	elif clr == 0x00FF00L
		isSending = True
	elif (clr & 0xFF) == 228
		foundIcon = True
	endIf
	x = x -1
	n = n -1
endWhile
if roomtree___pixelTestCondition == 0
	ptc = getTickCount() -ptc
	if ptc < 200
		roomtree___pixelTestCondition = 1
	else
		roomtree___pixelTestCondition = -1
	endIf
endIf
if isIdle
	flags = flags +"i"
elif foundIcon
	flags = flags +"o"
endIf
if isSending
	flags = flags +"s"
endIf
return flags
endFunction

int function roomtree___scrollTree(handle hwnd, object o, int childID, int reservedSpace)
; Scroll left/right the given tree as needed to make the following true
; regarding the MSAA rectangle's left edge (left):
;	- left is reservedSpace pixels or more right of the window's left edge.
;	- left is not right of the window's right edge.
var int left, int right, int top, int bottom
var int wleft = max(getWindowLeft(hwnd), 1)
var int wright = min(getWindowRight(hwnd), screenGetWidth())
var int safety = 500
var int scrollLeft, int scrollRight
while safety
	if !MSAA__rect(o, childID, left, right, top, bottom)
		return False
	endIf
	scrollLeft = (left -wleft < reservedSpace)
	scrollRight = (wright -left < 1)
	if scrollLeft && scrollRight
		return False
	endIf
	if scrollLeft
		;beep()
		util__WMScroll(hwnd, "LineLeft")
	elif scrollRight
		util__WMScroll(hwnd, "LineRight")
	else
		return True
	endIf
	safety = safety -1
endWhile
return False
endFunction

string function roomtree__serverName(handle hwnd)
; Returns the server name given any window in its hierarchy.
; The server name is the top-level tree node's name.
var collection tree = roomtree___treeRecord(getTopLevelWindow(hwnd))
if !tree
	return ""
endIf
var object o = tree.client
if !o
	return ""
endIf
var int childID = o.accNavigate(NavDir_FirstChild, 0)
if !childID
	; The tree is empty.
	; This catches "messages" caused by failed server connects.
	return roomtree___serverNameFromStatus(getTopLevelWindow(hwnd))
endIf
var string name = o.accName(childID)
; Abbreviate this, some server names are long.
; This cuts off at the first listed punctuation mark.
name = stringSegment(name, ".,?/;:!()[]", 1)
return name
endFunction

string function roomtree___serverNameFromStatus(handle hwnd)
; Return the name of the server for the indicated TeamTalk instance from its status line if possible.
; This applies during server connect attempts.
var string txt = ttutil__statusText(hwnd)
if !txt
	return ""
endIf
var int pos = stringContains(txt, "\"")
if !pos
	return ""
endIf
txt = stringChopLeft(txt, pos)
pos = stringContains(txt, "\"")
if !pos
	return ""
endIf
txt = stringLeft(txt, pos-1)
return txt
endFunction

;============================= End module roomtree =============================


;================================ Module ttutil ================================
; TeamTalk utility functions.
;
; Author:  Doug Lee

handle function ttutil__mainWindow(optional handle hwnd, int allowChats)
; Return the main top-level TeamTalk window for the focused or given window.
; If allowChats is True, returns the top-level window for a chat;
; otherwise returns null for such a case.
if !hwnd
	hwnd = getFocus()
endIf
hwnd = getTopLevelWindow(hwnd)
if !stringStartsWith(getOwningAppName(hwnd), "TeamTalk")
	return
endIf
if getParent(hwnd)
	; Happens when a dialog (e.g., volume adjustment) is active.
	hwnd = getParent(hwnd)
endIf
if ttutil__isMainWindow(hwnd)
	return hwnd
elif !allowChats
	return
endIf
; We already ruled out non-TeamTalk windows and non top-level windows.
; Dialogs are handled by the getParent check.
; Main windows returned earlier; so chats should be all we have left.
return hwnd
endFunction

int function ttutil__isMainWindow(handle hwnd)
; Returns True if and only if hwnd is the main top-level TeamTalk window.
if getWindowClass(hwnd) == "#32770"
&& (stringContains(getWindowName(hwnd), ttutil__scMainWindowName5)
|| stringContains(getWindowName(hwnd), ttutil__scMainWindowName4))
	return True
endIf
return False
endFunction

string function ttutil__statusText(handle hwnd)
; Return the status window text for the given TeamTalk instance.
hwnd = getTopLevelWindow(hwnd)
hwnd = findWindow(hwnd, "msctls_statusbar32", "")
if !hwnd
	return ""
endIf
; The status text in this app is all under childID 1.
var string txt = MSAA__windowName(hwnd, 1)
return txt
endFunction

;============================== End module ttutil ==============================


;================================ Module toolbar ===============================
; Module for reporting and tracking states of TeamTalk Classic toolbar checkbox states.
;
; Author:  Doug Lee
;

globals
; Structures by top-level TeamTalk window. Fields:
;	hTop: The top-level window (same as the key for this record).
;	hToolbar, oToolbar: The handle and MSAA client object for the toolbar window.
;	sIconNames: The names of the toolbar items (none appear in MSAA).
;	sIconStates: The states of the toolbar items (see toolbar___getStates()).
	collection toolbar___toolbars,
; How often toolbar__watchForChanges() watches for state changes,
; in tenths of a second.
	int toolbar___tenths,
; The schedule handle for toolbar__watchForChanges().
	int toolbar___sch

collection function toolbar__record()
; Get or make the record for the currently active client's toolbar.
var handle hwnd = getAppMainWindow(getFocus())
if !ttutil__isMainWindow(hwnd)
	return
endIf
var collection rec = chat4__winrecs__record(toolbar___toolbars, hwnd)
if !rec
	return
endIf

var handle hTop = hwnd
if !collectionItemExists(rec, "hTop")
	; New record.
	hwnd = findWindow(hwnd, "ToolbarWindow32", "")
	if !hwnd
		return
	endIf
	var object o = MSAA__objectFromWindow(hwnd)
	if !o
		return
	endIf
	var int n = o.accChildCount+0
	var string names
	if n == 12
		names = toolbar___msgNames45
	elif n == 10
		names = toolbar___msgNames43
	elif n == 9
		names = toolbar___msgNames42
	elif n == 8
		names = toolbar___msgNames41
	else
		; TODO:  Probably should report this.
		return
	endIf
	rec.hToolbar = hwnd
	rec.oToolbar = o
	rec.sIconNames = names
	rec.sIconStates = toolbar___getStates(rec)
	; Add this last as it signals record completion.
	rec.hTop = hTop
endIf

return rec
endFunction

string function toolbar___getStates(collection rec)
; Get the states of all toolbar items as a string.
var string states = ""
var object o = rec.oToolbar
var int n = o.accChildCount+0
var int i = 1
while i <= n
	states = states +"|" + intToString(o.accState(i))
	i = i +1
endWhile
return stringChopLeft(states, 1)
endFunction

int function toolbar__checkStates()
; Check and update the states of toolbar items,
; announcing any changes as needed.
; Returns True if there were any changes and False if not.
; Meant to be called on a schedule and/or after specific key presses.
var collection rec = toolbar__record()
if !rec
	return False
endIf
var string oldStates = rec.sIconStates
var string newStates = toolbar___getStates(rec)
if stringCompare(oldStates, newStates, True) == 0
	return False
endIf
var int n = stringSegmentCount(oldStates, "|")
if n != stringSegmentCount(newStates, "|")
	beep()
	rec.sIconStates = newStates
	return True
endIf
var int i = 1
var int oldState, int newState
while i <= n
	oldState = stringToInt(stringSegment(oldStates, "|", i)) & State_System_Checked
	newState = stringToInt(stringSegment(newStates, "|", i)) & State_System_Checked
	if newState != oldState
		toolbar___speakState(newState, i)
	endIf
	i = i +1
endWhile
rec.sIconStates = newStates
return True
endFunction

void function toolbar___speakState(int state, int i)
; Speak the given state.
var collection rec = toolbar__record()
var string name = stringSegment(rec.sIconNames, "|", i)
if stringCompare(name, "sep", False) == 0
	return
endIf
if stringContains(name, "/")
	if state
		name = stringSegment(name, "/", 2)
	else
		name = stringSegment(name, "/", 1)
	endIf
elif state
	name = formatString(toolbar___msgActive, name)
else
	name = formatString(toolbar___msgInactive, name)
endIf
sayUsingVoice(VCTX_Message, name, OT_Message)
endFunction

int function toolbar__watchForChanges(int tenths)
; Watch for changes in toolbar item states.
; Pass -1 to disable.
; Calls itself to do the watching.
if tenths
	; New schedule being established.
	if toolbar___sch
		unscheduleFunction(toolbar___sch)
		toolbar___sch = 0
	endIf
	toolbar___tenths = tenths
	if tenths > 0
		scheduleFunction("toolbar__watchForChanges", tenths)
	endIf
	return
endIf
; Internal call to do the watching.
toolbar___sch = scheduleFunction("toolbar__watchForChanges", toolbar___tenths)
toolbar__checkStates()
endFunction

void function toolbar__sayStatuses(int verbose)
; Speak the statuses of toolbar icons.
; If verbose is False, abbreviate the results.
var collection rec = toolbar__record()
var string states = toolbar___getStates(rec)
var int n = stringSegmentCount(states, "|")
var int i = 1
var int state, string name
var string buf = ""
var string sep = ", "
while i <= n
	state = stringToInt(stringSegment(states, "|", i)) & State_System_Checked
	if verbose
		toolbar___speakState(state, i)
	else
		name = stringSegment(rec.sIconNames, "|", i)
		if i == 1
			if state
				buf = buf +sep +toolbar___msgConnected
			else
				buf = buf +sep +toolbar___msgDisconnected
			endIf
		elif name != "sep" && state
			buf = buf +sep +name
		endIf
	endIf
	i = i +1
endWhile
if !verbose
	buf = stringChopLeft(buf, stringLength(sep))
	sayMessage(OT_User_Requested_Information, buf)
endIf
endFunction

;============================== End module toolbar =============================


;================================ Module tooltip ===============================
; Functions related to the management of tooltips.
;
; Author:  Doug Lee

string function tooltip__getText(/*?*/ handle hwnd, int x, int y)
; Fire the tooltip associated with the current or given screen location if possible, and return its text.
; If x and y are not passed or passed as 0, the active cursor's location is used.
; If hwnd is not passed or passed as 0, the window at the effective point is used.
hwnd = tooltip__getHandle(hwnd, x, y)
if !hwnd
	return ""
endIf
return getWindowText(hwnd, Read_Everything)
endFunction

handle function tooltip__getHandle(/*?*/ handle hwnd, int x, int y)
; Create and obtain the tooltip handle associated with the current or given screen location if possible.
; If x and y are not passed or passed as 0, the active cursor's location is used.
; If hwnd is not passed or passed as 0, the window at the effective point is used.
if !x && !y
	x = getCursorCol()
	y = getCursorRow()
endIf
if !hwnd
	hwnd = getWindowAtPoint(x, y)
endIf
var handle hwndDesk = util__getDesktopWindow()
if !hwndDesk
	return 0
endIf
var handle hwndBefore = getFirstChild(hwndDesk)
if !hwndBefore
	return 0
endIf
saveCursor()  JAWSCursor()  saveCursor()
moveTo(x, y)
winclick__send(hwnd, "hover", x, y)
pause()
var int cnt = 0
while getFirstChild(hwndDesk) == hwndBefore
	cnt = cnt +1
	pause()
	if cnt > 10
		return 0
	endIf
endWhile
var handle hwndAfter = getFirstChild(hwndDesk)
sayInteger(cnt)
return hwndAfter
endFunction

;============================== End module tooltip =============================


;================================== Module opt =================================
; User option code.
;
; Usage:
;	- Call opt__init() from autoStartEvent to set file and section names for options.
;	  This is optional; iff not done, getActiveConfiguration().ini and Options are used.
;	- Call opt__get() to get option values anywhere they are needed.
;	- Call opt__set() to alter option values.
;	- Call opt__cycle() from verbosity functions to handle the verbosity UI.
;
; Author:  Doug Lee

const
; A constant used to detect missing, as opposed to set-to-empty, .ini keys.
	opt___NoVal = 12345678

globals
; Information maintained by this module.
	collection opt___data

void function opt__init(/*?*/ string sectname, string filename)
; Set the section and file names used by this module for storing options.
if !opt___data
	opt___data = new collection
endIf
opt___setVal("sectname", sectname, "NonJCFOptions")
opt___setVal("filename", filename, getActiveConfiguration()+".jcf")
endFunction

void function opt___setVal(string name, string val, string dfl)
; Sets an internal configuration option to a given or default value.
; Helper for opt__init().
var string curval = opt___data[name]
if !val && !curval
	val = dfl
endIf
if val
	opt___data[name] = val
endIf
endFunction

variant function opt__get(string varName)
; Retrieves and returns the value of the indicated user-settable variable from the ini file for these scripts.
; This function makes sure to return integer 0 for the integer value 0, rather than returning the string "0" which shows up as True in a boolean test.
; varName: The name of the option to retrieve.
var
	string val
opt__init()
; Chop off descriptions.
varname = stringSegment(varname, ":", 1)
val = iniReadString(opt___data.sectname, varname, intToString(opt___NoVal), opt___data.filename)
; Favor integer 0 (False) rather than string "0" (True).
if stringContains(stringLeft(stringTrimLeadingBlanks(val), 1), "0")
	return ""
endIf
if val == opt___NoVal
	; All default values for options go here.
	if varname == SV_SpeakChatMessages
		val = "3"
	elif varname == SV_ChatMessageTimes
		val = "1"
	else
		val = ""
	endIf
endIf
return val
endFunction

int function opt__set(string varName, variant val)
; Sets the indicated user-settable variable in ttclassic.ini to the given value.
opt__init()
; Chop off descriptions.
varname = stringSegment(varname, ":", 1)
return iniWriteString(opt___data.sectname, varname, val, opt___data.filename)
endFunction

variant function opt__cycle(string varname, string choices, int iRetCurVal)
; Used by DlgSelectFunctionToRun cycler/toggler functions to cycle/toggle variable values.
; Varname should be the option name being handled (SV_* constant).
; Choices should be a list of text values, separated by vertical bars (|).
; iRetCurVal should be the integer taken by the calling toggler/cycler function.
; The return value of this function behaves according to specifications for DlgSelectFunctionToRun functions and should be returned directly by the toggler/cycler function.
; The integer value saved in ttclassic.ini will be the 0-based index of the chosen option.
var
	int i,
	int n
opt__init()
i = opt__get(varname)
n = string__segmentCount(choices, "|")
if !iRetCurVal
	; Update it.
	i = i +1
	if i >= n
		i = 0
	endIf
	opt__set(varname, i)
endIf
; i+1 rather than just i because i is 0-based and stringSegment is 1-based.
return stringSegment(choices, "|", i+1)
endFunction

;================================ End module opt ===============================


;================================ Module appOpt ================================
; User option code.

String Function jvoSpeakChatMessages (int RetCurValue)
return opt__cycle(SV_SpeakChatMessages, opt__msgSpeakChatChoices, RetCurValue)
EndFunction

String Function jvoChatMessageTimes (int RetCurValue)
return opt__cycle(SV_ChatMessageTimes, opt__msgOffOnChoices, RetCurValue)
EndFunction

void function appOpt___addVerbosityOption(string byRef lst, string optionData)
; Adds an option to the given verbosity (JAWS 8-) / option (JAWS 9+) list.
var
	string varAndDesc
varAndDesc = formatString("jvo%1:%2",
	stringSegment(optionData, ":", 1),
	stringSegment(optionData, ":", 2))
if lst
	lst = lst +"|"
endIf
lst = lst +varAndDesc
endFunction

Script AdjustJawsOptions ()
performScript adjustJAWSVerbosity()
endScript

Script AdjustJawsVerbosity ()
; Uses Jaws's central verbosity coding.
; Modified for skype usage by Ner
; To write more toggle functions for this, see toggle functions
; They *must* be done in the same manner.
var
	string lst

lst = ""
appOpt___addVerbosityOption(lst, SV_SpeakChatMessages)
appOpt___addVerbosityOption(lst, SV_ChatMessageTimes)
;This next section of the script and the supporting functions were written for JAWS 9.0 and later.  This also works in older JAWS versions because the version of JFW is checked first to determine which dlg to use. [TK, 2007-10-03]
if util__getJFWVersion() > 900000
	OptionsTreeCore(lst)
else
	JAWSVerbosityCore(lst)
endIf
EndScript

String Function NodeHLP (string NodeName)
	if NodeName == GetActiveConfiguration ()
return appOpt__msgNodeHlp
endIf
return nodeHlp(nodeName)
EndFunction

String Function jvoSpeakChatMessagesHlp()
return stringSegment(SV_SpeakChatMessages, ":", 3)
EndFunction

String Function jvoChatMessageTimesHlp()
return stringSegment(SV_ChatMessageTimes, ":", 3)
EndFunction

;============================== End module appOpt ==============================


;================================ Module apphelp ===============================
; Automatic access to custom application help (HTM) file.
;
; The help file should be named <configName>.htm, <configName> being the result of getActiveConfiguration().
; Alternatively, the file can be named by a HelpDocument key in a Script Options section of the active configuration's .jcf file.
; As a final alternative, the name can be supplied from autoStartEvent by calling apphelp__setDocName().
;
; Author:  Doug Lee
;
; This is public code.

globals
; Custom doc name if set by a call to apphelp__setDocName().
	string apphelp___docName,
; App in focus when the above name was set.
	string apphelp___docApp,
; Number of quick-succession calls to screenSensitiveHelp seen so far.
; apphelp___doHelp() also sets this to -1 to tell the screenSensitiveHelp script in this file to bypass itself and call up the chain.
	int apphelp___callCount

string function apphelp___getCustomHelpDoc()
; Gets the full path to the custom help document for the running scripts.
; Returns null if no such file is found.
var string docname let docname = ""
var string appname let appname = getActiveConfiguration()
if stringLength(apphelp___docApp)
&& stringCompare(appname, apphelp___docApp, True) == 0 then
	let docname = apphelp___docName
endIf
if !docName then
	var string fname let fname = getActiveConfiguration() +".jcf"
	let fname = findJAWSSettingsFile(fname)
	if stringLength(fname) && fileExists(fname) then
		let docname = iniReadString("Script Options", "HelpDocument", "", fname)
	endIf
endIf
if !docname then
	let docname = getActiveConfiguration()
	if !stringLength(docname) || docname == "default" then
		return ""
	endIf
	let docname = docname +".htm"
endIf
let docname = findJAWSSettingsFile(docname)
if !fileExists(docname) then
	return ""
endIf
return docname
endFunction

void function apphelp___doHelp()
; Implements the following behavior based on the number of JAWSKey+F1 presses and the presence or absence of a custom (.htm) help file:
;	One press: Normal ScreenSensitiveHelp in 0.6 seconds.
;	Two quick presses: The custom help document opens if present, or if not, JAWS app-specific (chm) help opens.
;	Three or more quick presses: JAWS-specific (chm) help opens if present.
var
	int callCount,
		string docname
let callCount = apphelp___callCount
let apphelp___callCount = 0
if callCount != 2 then
	; 1 = screenSensitiveHelp, 3+ = app-specific help.
	; The standard screenSensitiveHelp script will use isSameScript() to decide what to do.
	let apphelp___callCount = -1
	performScript screenSensitiveHelp()
	let apphelp___callCount = 0
	return
endIf

let docname = apphelp___getCustomHelpDoc()
if docname then
	if userBufferIsActive() then
		userBufferDeactivate()
	endIf
	sayMessage(OT_Message, apphelp___msgOpening)
	run("\"" +docname +"\"")
	return
endIf

; No custom help document, so revert to standard behavior.
let apphelp___callCount = -1
performScript screenSensitiveHelp()
let apphelp___callCount = 0
endFunction

script screenSensitiveHelp()
; Intercept that makes the apphelp___doHelp() function take effect.
if apphelp___callCount == -1 then
	; apphelp___doHelp sets apphelp___callCount to -1 to indicate it wants to call the original screenSensitiveHelp script.
	performScript screenSensitiveHelp()
	return
endIf

let apphelp___callCount = apphelp___callCount +1
scheduleFunction("apphelp___doHelp", 6)
endScript

void function apphelp__setDocName(string docname)
; Set the name of the custom help doc to load.
; Call from AutoStartEvent if necessary.
let apphelp___docApp = getActiveConfiguration()
let apphelp___docname = docname
endFunction

;============================== End module apphelp =============================


;================================ Module string ================================
; String manipulation functions.
;
; Author:  Doug Lee

string function string__segmentNext(string byRef s, string sep)
; Gets, returns, and removes from s the next string segment and the trailing separator if any.
; sep may be multi-character and is case sensitive.
; If sep is null, the entire string is treated as one segment.
var
	int pos,
	string s1
let pos = 0
if sep then
	let pos = stringContains(s, sep)
endIf
if !pos || !sep then
	let s1 = stringChopLeft(" " +s, 1)  ; avoids pointer duplication
	let s = ""
	return s1
endIf
let s1 = stringLeft(s, pos-1)
let s = stringChopLeft(s, pos +stringLength(sep) -1)
return s1
endFunction

int function string__segmentRemove(string byRef s, string sep, string r)
; Removes the first segment in s that exactly matches r, including case.
; Removes or leaves segment separators as appropriate.
; TODO: string__segmentRemove(s, "|", "val") is identical for s = "val" or "|val" or "val|":  the null string.
; Returns True if a segment is removed and False otherwise.
; Tested thoroughly for all segmentation and null cases 09/16/03.
var
	int slen, int seplen, int rlen, int pos
if !s || !sep || !r then
	return False
endIf

; Cases of s: (1) no r, (2) r, (3) r+sep+right, (4) left+sep+r, (5) left+sep1+r+sep2+right.
let slen = StringLength(s)
let seplen = StringLength(sep)
let rlen = stringLength(r)
let pos = stringContains(s, sep+r+sep)
if pos then
	; (5):  Most common case: r is in the middle of s somewhere, not at beginning or end.
	; This changes left+sep1+r+sep2+right to left+sep2+right.
	; pos currently points to the first character of sep1.
	let s = stringLeft(s, pos-1) +stringChopLeft(s, pos+seplen+rlen-1)
	return True
elif stringContains(s, r) == 1 then
	if slen == rlen then
		; (2) s is exactly r.
		let s = ""
		return True
	elif slen >= rlen+seplen && substring(s, rlen+1, seplen) == sep then
		; (3) s is r+sep+right (right may be null).
		let s = stringChopLeft(s, rlen+seplen)
		return True
	endIf
elif stringContains(stringRight(s, seplen+rlen), sep+r) then
	; (4):  s is left+sep+r (left may be null).
	let s = stringChopRight(s, seplen+rlen)
	return True
endif

; (1):  s does not contain r.
return False
endFunction

int function string__segmentCount(string s, string sep)
; Returns the number of segments in the given string.
; If sep is null, the result is 0.
; As of 03/15/03, the null string is defined to have zero segments (previously one).
; Any non-null string containing no separators has one segment.
; WARNING: Unlike the built-in (as of JAWS 7.0) stringSegmentCount function,
; this function counts the null string after a trailing delimiter;
; thus, string__segmentCount("a|b|", "|") is 3, where JAWS would say 2.
; Also, this function considers a multi-character sep as a multi-character delimiter,
; whereas stringSegmentCount allows each character of sep to be a delimiter.
; This function requires JAWS 7.0 or later.
var
	int len,
	int seplen,
	int l,
	int n
let seplen = stringLength(sep)
if seplen == 1 then
	let n = stringSegmentCount(s, sep)
	if stringRight(s, 1) == sep then
		; JAWS does not count the null string after a trailing delimiter, but I do.
		let n = n +1
	endIf
	return n
endIf
let len = stringLength(s)
if !len then
	return 0
endIf
if !seplen then
	return 0
endIf

; A little optimization made possible by JAWS 6.0's StringReplaceSubstrings function...
; Count segments by removing all separators and comparing string lengths.
let l = stringLength(string__ref(stringReplaceSubstrings(s, sep, "")))
let l = len -l
return (l / seplen) +1
endFunction

int function string__replace(string byRef s, string r1, string r2, string whatToDo)
; Replace r1 in s with r2 if r1 is found where whatToDo indicates.
; Note:  A space is sometimes prepended to and then removed from results in this function to keep JAWS from adding
; numeric strings arithmetically, which can happen in JAWS versions at least through 3.70.47 [DGL, 12/20/2000]
; whatToDo: l=left, r=right, a=anywhere, e=everywhere.
; Add t (for Trim) if whitespace should be removed at the point where s is split.
; Add c (for ConsiderCase) if case-sensitive matching is desired.
; For compatibility with older scripts, WhatToDo can also be an integer sum of flags:
; 0=l, 1=r, 2=a, 3=e, 8=t, 16=c.
; Returns the number of changes made to s, which must be passed by reference.
; If s is null, no changes are made and 0 is returned.
; If r1 is null, the following happens based on WhatToDo:
;	"e" - r2 is inserted between each character pair in s.  All blanks are removed first if "t" is included.
;	"l" or "a" - r2 is inserted after the first character of s, after blanks around it are removed if "t" is included.
;	"r" - r2 is inserted before the last character of s, after blanks around it are removed if "t" is included.
var
	string fromWhere,
	int trimEnds,
	int considerCase,
	int pos,
	int l1,
	string sa,
	string sb,
	int changed,
	int changeCount,
	int done

if !stringLength(s) then
	return 0
endIf

; Compatibility with older integer whatToDo parameter.
if stringLength(whatToDo) == 1 && stringContains(whatToDo, "0") then
	let whatToDo = "l"
	elif stringToInt(whatToDo) then
	let whatToDo = stringSegment("r|a|e|||||lt|rt|at|et|||||lc|rc|ac|ec|||||ltc|rtc|atc|etc", "|", stringToInt(whatToDo) & 31)
endIf

let whatToDo = stringLower(whatToDo)
let trimEnds = stringContains(whatToDo, "t")
let considerCase = stringContains(whatToDo, "c")
let fromWhere = whatToDo

; Special cases for null r1.
if !stringLength(r1) && fromWhere == "e" then
	if trimEnds then
		let sa = stringStripAllBlanks(s)
	else
		let sa = string__ref(s)
	endIf
	let s = ""
	let l1 = stringLength(sa)
	let pos = 1
	while pos < l1
		let s = string__cat(s, string__cat(substring(sa, pos, 1), r2))
		let pos = pos +1
	endWhile
	let s = string__cat(s, stringRight(sa, 1))
	return stringLength(sa)
elif !stringLength(r1) then
	if fromWhere == "l" || fromWhere == "a" then
		if trimEnds then
			let sb = stringTrimLeadingBlanks(s)
		else
			let sb = s +""
		endIf
		let sa = stringLeft(sb, 1)
		let sb = stringChopLeft(sb, 1)
		if trimEnds then
			let sb = stringTrimLeadingBlanks(sb)
		endIf
	else  ; fromWhere == "r"
		if trimEnds then
			let sb = stringTrimTrailingBlanks(s)
		else
			let sb = s +""
		endIf
		let sa = stringChopRight(sb, 1)
		if trimEnds then
			let sa = stringTrimTrailingBlanks(sa)
		endIf
		let sb = stringRight(sb, 1)
	endIf
	if !stringLength(sa) || !stringLength(sb) then
		return 0
	endIf
	let s = string__cat(sa, string__cat(r2, sb))
	return 1
endIf

let changeCount = 0
while !done
	let l1 = stringLength(r1)
	let changed = false
	if fromWhere == "l" then
		if !stringCompare(stringLeft(s, l1), r1, considerCase) then
			if trimEnds then
				let s = stringTrimLeadingBlanks(stringChopLeft(s, l1))
				let s = r2 + s
			else
				let s = r2 + stringChopLeft(s, l1)
			endIf
			let changed = true
		endIf
	elif fromWhere == "r" then
		if !stringCompare(stringRight(s, l1), r1, considerCase) then
			if trimEnds then
				let s = stringTrimTrailingBlanks(stringChopRight(s, l1))
				let s = s + r2
			else
				let s = stringChopRight(s, l1) + r2
			endIf
			let changed = true
		endIf
	else  ; fromWhere == "a" or "e"
		if considerCase then
			let pos = StringContains(s, r1)
		else
			let pos = StringContains(stringLower(s), stringLower(r1))
		endIf
		if pos then
			if trimEnds then
				let sa = stringTrimTrailingBlanks(stringLeft(s, pos-1))
				let sb = stringTrimLeadingBlanks(stringChopLeft(s, pos+l1-1))
				let s = " " +sa +r2 +sb
				let s = stringChopLeft(s, 1)
			else
				let s = " " +stringLeft(s, pos-1) +r2 +stringChopLeft(s, pos+l1-1)
				let s = stringChopLeft(s, 1)
			endIf
			let changed = True
		endIf
	endIf
	if fromWhere == "e" then
		let done = !changed
	else
		let done = True
	endIf
	if changed then
		let changeCount = changeCount +1
	endIf
endWhile
return changeCount
endFunction

string function string__join(string s1, string sep, string s2)
; Returns the two given strings joined with the given separator.
; If s1 or s2 is empty, the separator is not included.
if !s1 then
	return s2
elif !s2 then
	return s1
else
	return s1 +sep +s2
endIf
endFunction

string function string__ref(variant v)
; Type cast from anything to string.
; Useful for allowing object results to be included in string concatenations without a compile-time error.
return v
endFunction

string function string__cat(string s1, string s2)
; Safe string concatenator.
return stringChopLeft(" " +s1 +s2, 1)
endFunction

;============================== End module string ==============================


;================================= Module util =================================
; Miscellaneous utility functions
;
; Author:  Doug Lee

int function util__compare(variant v1, variant v2, string how)
; Comparison that returns 1 if v1 is greater than v2, 0 if equal to v2, and -1 if less than v2.
; Comparison types (values of how):
;	n - Numeric comparison (non numbers sort as 0).
;	i - Case-insensitive string comparison.
;	s (default if how is null) - Case-sensitive string comparison.
;	<funcName> - Returns funcname(v1, v2)
;	<funcCall> - Evaluates funcCall substituting %1 with "v1" and %2 with "v2."
; TODO:  Beeps and returns 0 in the following cases:
;	- String comparison (how == i or s) for JAWS versions older than 6.20.
;	- Function calls (funcName or funcCall) when v1 or v2 contains a quote (").
; This function requires JAWS 6.20 or later.
if !how then
	let how = "s"
endIf
if stringLength(how) == 1 then
	if how == "n" then
		if stringToInt(v1) == stringToInt(v2) then
			return 0
		elif stringToInt(v1) < stringToInt(v2) then
			return -1
		else
			return 1
		endIf
	elif how == "s" || how == "i" then
		return stringCompare(v1, v2, (how == "s"))
	endIf
endIf
if stringContains(v1, "\"") || stringContains(v2, "\"") then
	; Incompatible with formatStringWithEmbeddedFunctions().
	beep()
	return 0
endIf
if stringContains(how, "(") then
	string__replace(how, "%1", "\""+string__ref(v1)+"\"", "e")
	string__replace(how, "%2", "\""+string__ref(v2)+"\"", "e")
	return stringToInt(formatStringWithEmbeddedFunctions("<" +how +">"))
endIf
return stringToInt(formatStringWithEmbeddedFunctions("<" +how +"(\"" +string__ref(v1) +"\", \"" +string__ref(v2) +"\")>"))
endFunction

int function util__abs(int n)
; Return the absolute value of n.
if n < 0 then  return 0-n
else return n
endIf
endFunction

int function util__isSpecialFocus()
; Returns True if there is a special condition for which normal focus handling should not apply.
; Currently, the conditions that are considered special are menusActive, userBufferIsActive, inHJDialog, and switchingTasks (Alt+Tab is active).
return menusActive() ||userBufferIsActive() ||inHJDialog() || util__switchingTasks()
endFunction

int function util__switchingTasks()
; Returns True if the user is currently Alt+Tabbing between tasks.

; This approach is adapted from JAWS 9.0.2169's default.jss FocusChangedEventEx function.
if getWindowClass(getFocus()) == "#32771"
&& getObjectTypeCode()== WT_LISTBOXITEM then
	return True
endIf
return False
endFunction

handle function util__getDesktopWindow()
; Analogous to the Windows call of the same name.
var
	handle hwnd,
	string hlist, int hcount, int i

; Switching to the invisible cursor keeps GetWindowTypeCode from returning 3 (WT_Edit) for all windows when the virtual cursor is active.
saveCursor() invisibleCursor()

; First try handles known to be the Desktop window in various versions of Windows.
; 128=95/98, 65566=NT, 65556=XP, 65546=2k.
; Thanks to Victor Tsaran for helping collect these handles. [DGL, 04/28/03]
; 65552 for Windows 8.1.
let hlist = "128|65566|65556|65552|65546|65536|65526|65516|65506"
let hcount = 8
let i = 1
while i <= hcount
	let hwnd = stringToHandle(stringSegment(hlist, "|", i))
	if getWindowTypeCode(hwnd) == WT_Desktop then
		return hwnd
	endIf
	let i = i +1
endWhile

; The hard way...

; Start at the focused window, or if there isn't one, from an arbitrary window on the screen.
let hwnd = getFocus()
if !hwnd then
	let hwnd = getWindowAtPoint(300, 300)  ; A valid point regardless of screen resolution
endIf
let hwnd = getFirstWindow(getTopLevelWindow(hwnd))
while hwnd
	if getWindowTypeCode(getParent(hwnd)) == WT_Desktop then
		return getParent(hwnd)
	endIf
	let hwnd = getNextWindow(hwnd)
endWhile
return 0
endFunction

string function util__callStackInfo(int nlevels, string flags, /*?*/ int skip)
; Produce a representation of the current call stack formatted according to the given parameters.
; The following shortenings are applied to each script/function call in the return value:
;	- The shared folder path is replaced with "shared."
;	- The user folder path is removed.
;	- The word "function" is removed.
;	- The word "script" is replaced with "$."
; Requires JAWS 10 or later because the JAWS 10+ getScriptCallStack() is required.
; nlevels: Number of levels from starting position up the stack to print.  1 means just current function.  0 or less means everything.
; flags: f for forward order (default is reverse), s for short (one line and no parameters).
; skip: How many calls to skip over besides this function itself, default 0.
; Returns the requested call stack information or null if the JAWS version is too old.
var
	string userFolder, string sharedFolder,
	int short, int forward,
	int stacksize
if util__getJFWVersion() < 1000000 then
	return ""
endIf
let flags = stringLower(flags)
let short = stringContains(flags, "s")
let forward = stringContains(flags, "f")
let userFolder = getJAWSSettingsDirectory()
let sharedFolder = getSharedSettingsDirectory()
var collection stack = util__callStack()
var int i let i = 2  ; skip this function's entry.
let i = i +skip
let stacksize = stack.count
if nlevels > 0 then
	let nlevels = min(nlevels, stacksize-i+1)
else
	let nlevels = stacksize-i+1
endIf
if nlevels <= 0 then
	return ""
endIf
var int delta let delta = 1
if forward then
	let delta = -1
	let i = i +nlevels -1
endIf
var string buf let buf = ""
while nlevels
	if buf then
		if forward then
			let buf = buf +" -> "
		else
			let buf = buf +" <- "
		endIf
	endIf
	var collection  entry = stack[intToString(i)]
	var string fname = entry.file
	let fname = stringReplaceSubstrings(fname, sharedFolder, "")
	let fname = stringReplaceSubstrings(fname, userFolder, "user")
	if stringRight(fname, 4) == ".jsb" then
		let fname = stringChopRight(fname, 4)
	endIf
	let buf = buf +fname +" "
	if entry.type == "script" then
		let buf = buf +"$"
	endIf
	if short then
		let buf = buf +stringSegment(entry.call, "(", 1)
	else
		let buf = buf +string__ref(entry.call)
	endIf
	let i = i +delta
	let nlevels = nlevels -1
endWhile
return buf
endFunction

globals object util___csRE
collection function util__callStack()
; Return a collection of collections representing the script call stack.
; Requires JAWS 11 update 2 or later and the VBScript.RegExp object.
; Index the result by stringified 1-based integers.
; The most recent call is item 1.
; Structure of an item:
;	file: The file from which the script/function came, either "builtin" or a full path.
;	type: The type ("script" or "function", lower case).
;	call: The full call with parameters.
if !util___csRE then
	let util___csRE = util__createObject("VBScript.RegExp")
	if !util___csRE then
		return
	endIf
	let util___csRE.Global = -1
	let util___csRE.IgnoreCase = 0
	let util___csRE.Pattern = "\10([^\10\t]*)\t(script |function )"
endIf
var string sStack let sStack = "\10" +getScriptCallStack()
var object matches let matches = util___csRE.execute(sStack)
var object match
var int n let n = matches.count
var int i let i = 1  ; Skip this function itself.
var string tmp, int pos1, int pos2
var collection stack let stack = new collection
var collection entry
stack.count = 0
while i < n
	entry = new collection
	stack.count = stack.count +1
	stack[intToString(stack.count)] = entry
	let match = matches(i)
	; Skip the \10 here.
	; Also switch from 0-base to 1-base.
	let pos1 = match.firstIndex +2
	if i == n-1 then
		let pos2 = stringLength(sStack) +1
	else
		let pos2 = matches(i+1).firstIndex +1
	endIf
	let tmp = substring(sStack, pos1, pos2-pos1)
	entry.file = string__segmentNext(tmp, "\t")
	entry.type = string__segmentNext(tmp, " ")
	entry.call = tmp
	let i = i +1
endWhile
return stack
endFunction

globals
	int util___JAWSVer
int function util__getJFWVersion()
; GetJFWVersion with caching to make it faster.
; Also includes the build number even when GetJFWVersion() does not.
var
	string js
if !util___JAWSVer then
	let util___JAWSVer = getJFWVersion()
	let js = intToString(util___JAWSVer)
	if stringRight(js, 3) == "000" then
		let js = util__getJAWSInfo()
		let js = stringSegment(js, "|", 1)
		let js = stringSegment(js, ".", 3)
		let util___JAWSVer = util___JAWSVer +stringToInt(js)
	endIf
endIf
return util___JAWSVer
endFunction

string function util__getJAWSInfo()
; Returns a list of fields, separated by "|", as follows:
; 1.  The JAWS version number from the JFWUI2 window, in the form [M]M.mm.bbb; example, 5.00.584.
; 2.  The JAWS serial number from the JFWUI2 window.
; 3.  The JAWS serial number from GetJFWSerialNumber().
; The build number returned by this function is accurate, whereas it is "000" in GetJFWVersion in older JAWS versions.
var
	handle hwnd0, handle hwnd,
	int pos,
	string sn, string ver

; Find the JAWS dialog, which exists even when invisible or when JAWS is on the system tray.
let hwnd0 = getFirstChild(findTopLevelWindow("JFWUI2", "JAWS"))
if !hwnd0 then
	return ""
endIf

; Get the serial number from the status bar's window name.
let hwnd = getNextWindow(hwnd0)
let sn = stringLower(getWindowName(hwnd))
let pos = stringContains(sn, "ber")
if pos then
	let sn = intToString(stringToInt(substring(sn, pos+4, 10)))
else
	let sn = ""
endIf

; Find the static containing the version number and get it.
let hwnd = getFirstChild(hwnd0)
let ver = ""
while hwnd && !ver
	let ver = stringTrimTrailingBlanks(getWindowName(hwnd))
	let pos = stringContainsFromRight(ver, " ")
	if pos then
		let ver = stringChopLeft(ver, pos)
	else
		let ver = ""
	endIf
	let hwnd = getNextWindow(hwnd)
endWhile

; The above version retrieval sometimes fails, so try the normal way.
if !ver then
	ver = intToString(getJFWVersion())
	var int verlen = stringLength(ver)
	ver = formatString("%1.%2.%3",
		stringChopRight(ver, 5),
		substring(ver, verlen-4, 1),
		stringRight(ver, 4)
	)
endIf

if !sn && !ver then
	return ""
else
	return ver +"|" +sn +"|" +getJFWSerialNumber()
endIf
endFunction

object function util__createObject(string objname)
; A CreateObject wrapper that avoids causing 5-7 second delays in DOS boxes when called if JAWS is running as a service.
; As of June 11, 2011, uses the CoCreateInstance method of creating an object
; unless that doesn't work, whereupon the GetObjectHelper method is used.
; Previously the CoCreateInstance method was only used if JAWS was running as a service.
var object o let o = createObjectEx(objname, False)
if !o then
	let o = createObject(objname)
endIf
return o
endFunction

int function util__WMScroll(handle hwnd, string cmd)
; Scroll the given window or get info about its scroll position according to cmd.
; This function uses WinAPI messages to accomplish scrolling.
; Commands allowed (18):
;	LineLeft/Right/Up/Down, PageLeft/Right/Up/Down.
;	Left/Top/Right/Bottom.
;	H/VThumbPosition/Track.
;	H/VEndScroll.
var
	int msg, int wParam
; Translate cmd in place into something more directly parseable - a two-character code:
;	h/v:  Horizontal or vertical operation.
;	-+<>^$pte:
;		-+<>:  Line or page forward/back.
;		^$:  Fully left/top or right/bottom.
;		pte:  ThumbPosition, ThumbTrack, or EndScroll.
; This system avoids up to 18 string comparisons and guarantees that cmd is valid.
if !util__stringTran(cmd, False,
		" LineLeft LineRight LineUp LineDown PageLeft PageRight PageUp PageDown Left Top Right Bottom HThumbPosition VThumbPosition HThumbTrack VThumbTrack HEndScroll VEndScroll",
		" h- h+ v- v+ h< h> v< v> h^ v^ h$ v$ hp vp ht vt he ve"
) then
	return False
endIf

; Now set msg and wParam.
if stringLeft(cmd, 1) == "h" then
	let msg = 0x0114  ; WM_HScroll
else
	let msg = 0x0115  ; WM_VScroll
endIf
; This produces an SB_* constant from winuser.h.
let wParam = stringContains("-+<>pt^$e", stringRight(cmd, 1)) -1
return sendMessage(hwnd, msg, wParam, 0)
endFunction

int function util__stringTran(string byRef val0, int caseSensitive, string s1, string s2)
; Translate val in place by finding it in s1 and replacing it with the corresponding s2 value.
; Returns True on success and False on failure.
; TODO: This function probably belongs in the String module.
var
	string val,
	string sep1, string sep2,
	int idx
let val = val0
if !caseSensitive then
	let val = stringLower(val)
	let s1 = stringLower(s1)
endIf
let sep1 = stringLeft(s1, 1)
let sep2 = stringLeft(s2, 1)
let idx = stringSegmentIndex(s1, sep1, val, True)
if !idx || idx > stringSegmentCount(s2, sep2) then
	return False
endIf
let val0 = stringSegment(s2, sep2, idx)
return True
endFunction

variant function util__vcast(variant v)
; Cast any type to variant.
return v
endFunction

;=============================== End module util ===============================


;=============================== Module winclick ===============================
; Support for sending mouse clicks of various sorts directly to a window.
; This permits clicking on obscured windows
; or clicking without checking for what is in the foreground.
;
; Author:  Doug Lee, with credit to Jim Bauer for the idea and some initial research.
;
; TODO:
;	- No support for mouse wheel events.
;	- Alt+clicks not supported.
;	- Non-client clicks not supported, such as for scrollbar elements.


int function winclick__send(handle hwnd, string which, /*?*/ int x, int y)
; Send a click to the given window using a sendMessage() call.
; hwnd: The window to which to send an event.
; which: Which event and keys to include.  Click specifiers:
;	Left, Right, or Middle:  Which button to use for a click.
;	dbl (optional):  Include if you want a double click.
;	Click (optional):  Not required.
;	Hover: Use in place of the above to cause a hover event instead of a click.
;	Leave: Use in place of the above to cause a leave event.
; Examples: leftDblClick, leftClick, leftDbl, left, hover.
; Key specifiers (include zero or more):
;	ctrl or control: A control key is down.
;	shift: A shift key is down.
; x: The x coordinate to click or hover over (optional).
; y: The y coordinate to click or hover over (optional).
; If x and y are not passed or are 0,
; the top left corner of the Window's client area is used.

; Returns the sendMessage() result.
return winclick___doClick(True, hwnd, which, x, y)
endFunction

void function winclick__post(handle hwnd, string which, /*?*/ int x, int y)
; Send a click to the given window using a sendMessage() call.
; hwnd: The window to which to send an event.
; which: Which event and keys to include.  Click specifiers:
;	Left, Right, or Middle:  Which button to use for a click.
;	dbl (optional):  Include if you want a double click.
;	Click (optional):  Not required.
;	Hover: Use in place of the above to cause a hover event instead of a click.
;	Leave: Use in place of the above to cause a leave event.
; Examples: leftDblClick, leftClick, leftDbl, left, hover.
; Key specifiers (include zero or more):
;	ctrl or control: A control key is down.
;	shift: A shift key is down.
; x: The x coordinate to click or hover over (optional).
; y: The y coordinate to click or hover over (optional).
; If x and y are not passed or are 0,
; the top left corner of the Window's client area is used.

winclick___doClick(False, hwnd, which, x, y)
endFunction

int function winclick___doClick(int useSend, handle hwnd, string which, /*?*/ int x, int y)
; Send a click to the given window using a sendMessage() or postMessage call.
; See winclick__sendClick() or winclick__postClick() for details on usage.
; Returns the sendMessage() result if sendMessage() is used, or nothing if postMessage() is used.
; Fine print: This function sends two messages, ; one for buttonDown and one for buttonUp.
; The sendMessage result comes from the first of these calls when sendMessage is used.
; Only one message is sent for a hover or leave.
let which = stringLower(which)
var int msgDown, int msgUp
var int wParam let wParam = 0
if stringContains(which, "hover") then
	let msgDown = 0x02A1  ; WM_MOUSEHOVER
	let msgUp = 0  ; skip that one.
elif stringContains(which, "leave") then
	let msgDown = 0x02A3  ; WM_MOUSELEAVE
	let msgUp = 0  ; skip that one.
elif stringContains(which, "left") then
	if stringContains(which, "dbl") then
		let msgDown = 0x0203  ; WM_LBUTTONDBLCLK
	else
		let msgDown = 0x0201  ; WM_LBUTTONDOWN
	endIf
	let msgUp = 0x0202  ; WM_LBUTTONUP
	let wParam = wParam | 1  ; MK_LBUTTON
elif stringContains(which, "right") then
	if stringContains(which, "dbl") then
		let msgDown = 0x0206  ; WM_RBUTTONDBLCLK
	else
		let msgDown = 0x0204  ; WM_RBUTTONDOWN
	endIf
	let msgUp = 0x0205  ; WM_RBUTTONUP
	let wParam = wParam | 2  ; MK_RBUTTON
elif stringContains(which, "middle") then
	if stringContains(which, "dbl") then
		let msgDown = 0x0209  ; WM_MBUTTONDBLCLK
	else
		let msgDown = 0x0207  ; WM_MBUTTONDOWN
	endIf
	let msgUp = 0x0208  ; WM_MBUTTONUP
	let wParam = wParam | 0x10  ; MK_MBUTTON
else
	beep()
	return False
endIf
if stringContains(which, "shift") then
	let wParam = wParam | 4  ; MK_SHIFT
endIf
if stringContains(which, "ctrl") || stringContains(which, "control") then
	let wParam = wParam | 8  ; MK_CONTROL
endIf
var int lParam let lParam = 0
if x || y then
	if !msgUp then
		saveCursor()  JAWSCursor()  saveCursor()
		moveTo(x, y)
	endIf
	var int left, int right, int top, int bottom
	; -4 is ObjID_Client
	if MSAA__windowRect(hwnd, -4, left, right, top, bottom) then
		let x = x -left
		let y = y -top
	else
		let x = x -getWindowLeft(hwnd)
		let y = y -getWindowTop(hwnd)
	endIf
	let lParam = makeLong(x, y)
endIf
var int result
if useSend then
	let result = sendMessage(hwnd, msgDown, wParam, lParam)
	if msgUp then
		sendMessage(hwnd, msgUp, 0, lParam)
	endIf
else
	postMessage(hwnd, msgDown, wParam, lParam)
	if msgUp then
		postMessage(hwnd, msgUp, 0, lParam)
	endIf
endIf
if !msgUp then
	pause()
endIf
return result
endFunction

;============================= End module winclick =============================


;================================= Module MSAA =================================
; Miscellaneous functions for MSAA access
;
; Author:  Doug Lee

string function MSAA__windowName(handle hwnd, /*?*/ int childID, int objID)
; Get the MSAA name for the given window.
return MSAA__windowProp(hwnd, "name", childID, objID)
endFunction

int function MSAA__windowState(handle hwnd, /*?*/ int childID, int objID)
; Get the MSAA state flags for the given window.
return MSAA__windowProp(hwnd, "state", childID, objID)
endFunction

int function MSAA__windowRect(handle hwnd, int objID, int byRef left, int byRef right, int byRef top, int byRef bottom)
; Return by reference the rectangle for the given object ID in the given window.
; Returns True on success and False on failure.
let left = 0
let right = 0
let top = 0
let bottom = 0
var object o, int childID
let o = getObjectFromEvent(hwnd, objID, 0, childID)
if !o then
	return False
endIf
if !MSAA__rect(o, 0, left, right, top, bottom) then
	return False
endIf
return True
endFunction

variant function MSAA__windowProp(handle hwnd, string whichProp, /*?*/ int childID, int objID)
; Returns any one of several possible MSAA properties for the given window.
; Supported properties:
;	Name, Role, State, Value.
;	Description, KeyboardShortcut, Help, ChildCount.
var
	variant result,
	object o
if whichProp == "childCount" || whichProp == "role" || whichProp == "state" then
	let result = util__vcast(0)
else
	let result = util__vcast("")
endIf
if !hwnd then
	return result
endIf
let o = MSAA__objectFromWindow(hwnd, objID)
if !o then
	return result
endIf
let childID = childID +0
if whichProp == "name" then
	return o.accName(childID)
elif whichProp == "state" then
	return o.accState(childID)
elif whichProp == "role" then
	return o.accRole(childID)
elif whichProp == "value" then
	return o.accValue(childID)
elif whichProp == "description" then
	return o.accDescription(childID)
elif whichProp == "help" then
	return o.accHelp(childID)
elif whichProp == "keyboardShortcut" then
	return o.accKeyboardShortcut(childID)
elif whichProp == "childCount" then
	if childID then
		return o.accChild(childID).accChildCount
	else
		return o.accChildCount
	endIf
endIf
return result
endFunction

int function MSAA__rect(object o, int childID, int byRef left, int byRef right, int byRef top, int byRef bottom)
; Get the bounding rectangle for the given MSAA object and childID.
; The rectangle is returned 1-based, as JAWS likes its rectangles.
; MSAA provides 0-based left/top/width/height.
if !o then
	let left = 0
	let right = 0
	let top = 0
	let bottom = 0
	return False
endIf
if !childID then
	let childID = 0
endIf
o.accLocation(intRef(left), intRef(top), intRef(right), intRef(bottom), childID)
if !left && !right && !top && !bottom then
	return False
endIf
; but we want 1-based left/right/top/bottom.
let left = left +1
let top = top +1
let right = left +right
let bottom = top +bottom
return True
endFunction

globals
	handle MSAA___lastHwnd,
	int MSAA___lastObjID,
	object MSAA___lastObj

object function MSAA__objectFromWindow(handle hwnd, /*?*/ int objID)
; Return the requested (default ObjID_Client) MSAA object from the given window.
if !stringLength(util__vcast(objID)) then
	let objID = -4  ; ObjID_Client
endIf
if hwnd == MSAA___lastHwnd && objID == MSAA___lastObjID && MSAA___lastObj then
	return MSAA___lastObj
endIf
var object o, int childID
let o = getObjectFromEvent(hwnd, objID, 0, childID)
if !o then
	return o
endIf
let MSAA___lastHwnd = hwnd
let MSAA___lastObjID = objID
let MSAA___lastObj = o
return o
endFunction

;=============================== End module MSAA ===============================


;================================ Module compat ================================
; Compatibility shim to make modern scripts run on older JAWS versions.
; When this module supports a function, it supports it back through JAWS 11.0 unless otherwise indicated.
; Functions begin with comment lines indicating the function's classification:
;	Override:  An override (Unknown() should be the only one).
;	Support:  Support code for other functions.
;	NativeStart:  A JAWS function replacement for JAWS 11.0 through the version just before the one indicated on the comment line.
;
; To add a new function catcher:
;	- Add it in the If block in Unknown().
;	- Write compat_<funcName> after the fashion of the others in this module.
;
; Author:  Doug Lee


globals
	int compat___JAWSVer

int function compat___getJFWVersion()
; Support
; Used internally to speed recognition of the current JAWS version by caching the getJFWVersion() result.
if !compat___JAWSVer
	compat___JAWSVer = getJFWVersion()
endIf
return compat___JAWSVer
endFunction

string function compat___StringRef(variant v)
; Support
; Type cast from anything to string.
; Useful for allowing object results to be included in string concatenations without a compile-time error.
return v
endFunction

variant function unknown(string theName, int isScript,
	variant byRef v1, variant byRef v2, variant byRef v3, variant byRef v4, variant byRef v5,
	variant byRef v6, variant byRef v7, variant byRef v8, variant byRef v9, variant byRef v10
)
; Override
; Hander-offer for functions that are missing in the running JAWS.
if isScript
	return unknown(theName, isScript, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10)
endIf
var int jver = compat___getJFWVersion()
if jver >= 1400000 && v1
	; Indicates recursion; don't mess with those.
	; This prevents, for instance, a recursive stringStartsWith() call
	; from getting sent incorrect arguments below in JAWS 14+.
	return unknown(theName, isScript, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10)
endIf
if stringCompare(theName, "stringStartsWith", False) == 0
	return compat___stringStartsWith(v1, v2)
elif stringCompare(theName, "collectionItemCount", False) == 0
	return compat___collectionItemCount(v1)
elif stringCompare(theName, "collectionItemExists", False) == 0
	return compat___collectionItemExists(v1, v2)
elif stringCompare(theName, "collectionRemoveItem", False) == 0
	return compat___collectionRemoveItem(v1, v2)
elif stringCompare(theName, "collectionRemoveAll", False) == 0
	return compat___collectionRemoveAll(v1)
elif stringCompare(theName, "collectionCopy", False) == 0
	return compat___collectionCopy(v1)
endIf
return unknown(theName, isScript, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10)
endFunction

int function compat___stringStartsWith(string s1, string s2)
; NativeStart: 13.00
var int len1 = stringLength(s1)
var int len2 = stringLength(s2)
if len1 < len2
	return False
endIf
s1 = stringLeft(s1, len2)
return (stringCompare(s1, s2, False) == 0)
endFunction

int function compat___collectionItemCount(collection c)
; NativeStart: 14.00
var string k
var int cnt = 0
forEach k in c
	cnt = cnt +1
endForEach
return cnt
endFunction

int function compat___collectionItemExists(collection c, string k0)
; NativeStart: 12.00
var variant null
if c[k0]
	; Catches non-empty values.
	return True
endIf
; Now to catch empty values.
var string k
forEach k in c
	if stringCompare(k, k0, False) == 0
	&& c[k] != null
		return True
	endIf
endForEach
return False
endFunction

void function compat___collectionRemoveAll(collection c)
; NativeStart: 12.00
var variant null
var string k
forEach k in c
	c[k] = null
endForEach
endFunction

int function compat___collectionRemoveItem(collection c, string k)
; NativeStart: 12.00
var variant null
if !compat___collectionItemExists(c, k)
	return False
endIf
c[k] = null
return True
endFunction

int function compat___collectionCopy(collection c)
; NativeStart: 12.00
; ToDo: This is not a deep copy and so will not always work as expected.
var collection nc nc = new collection
var string k
forEach k in c
	nc[k] = c[k]
endForEach
return nc
endFunction

;============================== End module compat ==============================

