package org.soyatec.connector.api.i2c;

import org.eclipse.jface.dialogs.MessageDialog;

import org.eclipse.swt.widgets.Display;

import org.soyatec.connector.api.IConnection;
import org.soyatec.connector.api.IConnectionManager;
import org.soyatec.connector.api.i2c.jni.II2cDllByJNA;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;


public class I2cConnection implements IConnection<II2cSettings>, Runnable {
    static int timeout = 1000;
    private boolean connected;
    private static final int DEFAULT_CHANNEL_ID = 0;
    private static int baudRate;
    private static int deviceAddress;
    private String name;
    private II2cSettings i2cSettings;

    /**
     * which the received data will be written to, if the {@link #autoReceive}
     * is true.
     */
    private boolean autoReceive = false;

    public I2cConnection(II2cSettings i2cSettings) throws IOException {
        init(i2cSettings);
        new Thread(this).start();
    }
    
    private void init(II2cSettings i2cSettings){
    	this.setSettings(i2cSettings);
    }

    /**
     * This function opens the indexed channel and returns a handle to it.
     *
     * @param index
     *            Index of the channel,0 or 1.
     * @param baudrate
     *            Set the I2C baud rate.
     * @return
     */
    public static I2cConnection createConnection(II2cSettings ii2cSettings)
        throws IOException {
        I2cConnection i2cConnector = new I2cConnection(ii2cSettings);
        String baudRateStr = ii2cSettings.getfBaudRate();
        String deviceAddressStr = ii2cSettings.getfDeviceAddress();
        baudRate = I2cUtils.getbaudRateValue(baudRateStr);
        deviceAddress = I2cUtils.convertStringToInt(deviceAddressStr);

        // int openResult = JNativeI2cDLL.INSTANCE.i2cOpenChannel(
        // DEFAULT_CHANNEL_ID, baudRate);
        try {
            int openResult = II2cDllByJNA.instance.I2C_OpenChannel(DEFAULT_CHANNEL_ID,
                    baudRate);

            if (I2cUtils.checkI2cReturn(openResult)) {
                i2cConnector.connected = true;
            } else {
                i2cConnector.connected = false;
            }
        } catch (Exception e) {
            i2cConnector.connected = false;
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });
            e.printStackTrace();
        } catch (Error e) {
            i2cConnector.connected = false;
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });
            e.printStackTrace();
        }

        return i2cConnector;
    }

    /**
     * Closes a channel and frees all resources that were used by it..
     */
    public void close() {
        setConnectionManager(null);
        connected = false;

        try {
            int closeResult = II2cDllByJNA.instance.I2C_CloseChannel(0);
            I2cUtils.checkI2cReturn(closeResult);
        } catch (Exception e) {
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });
        } catch (Error e) {
            connected = false;
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });
            e.printStackTrace();
        }
    }

    @Override
    public void run() {
    }

    public boolean isAutoReceive() {
        return autoReceive;
    }

    public void setAutoReceive(boolean autoReceive) {
        this.autoReceive = autoReceive;
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public II2cSettings getSettings() {
        return this.i2cSettings;
    }

    @Override
    public void setSettings(II2cSettings settings) {
        this.i2cSettings = settings;
    }

    @Override
    public void setName(String name) {
        this.name = name;
    }

    @Override
    public boolean updateSettings(II2cSettings settings) {
        return false;
    }

    @Override
    public void setConnectionManager(IConnectionManager connectionManager) {
        
    }

    @Override
    public void send(String data) throws IOException {
    }

    @Override
    public void send(char data) throws IOException {
    }

    /**
     * This function writes the specified number of bytes from an addressed I2C
     * slave.
     *
     * @param deviceAddress
     *            Address of the I2C slave
     * @param sizetowrite
     *            Number of bytes to be written
     * @param inputbuffer
     *            Pointer to the buffer from where data is to be written
     * @param sizetoread
     *            Number of bytes to be read
     * @param outputbuffer
     *            Number of bytes to be read
     * @return
     * @throws IOException
     */
    public boolean sendI2cCmd(int sizetowrite, byte[] inputbuffer,
        int sizetoread, byte[] outputbuffer) throws IOException {
        int setCmdResult = -1;
        int sendCount = 1;

        try {
            while ((setCmdResult != 0) && (sendCount < 10)) {
                setCmdResult = II2cDllByJNA.instance.I2C_DeviceStream(DEFAULT_CHANNEL_ID,
                        deviceAddress, sizetowrite, inputbuffer, sizetoread,
                        outputbuffer);
                System.out.println("setCmdResult: " + setCmdResult);
                sendCount++;
            }
        } catch (Exception e) {
            e.printStackTrace();
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });

            return false;
        } catch (Error e) {
            connected = false;
            Display.getDefault().syncExec(new Runnable() {
                    @Override
                    public void run() {
                        MessageDialog.openWarning(Display.getDefault()
                                                         .getActiveShell(),
                            Messages.Message_INFO, Messages.Message_FT_I2C);
                    }
                });
            e.printStackTrace();
        }

        return I2cUtils.checkI2cReturn(setCmdResult);
    }

    @Override
    public OutputStream getRemoteOutputStream() {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public boolean isConnected() {
        return connected;
    }

    @Override
    public InputStream getRemoteInputStream() {
        // TODO Auto-generated method stub
        return null;
    }

    public static int getBaudRate() {
        return baudRate;
    }

    public static int getDeviceAddress() {
        return deviceAddress;
    }

    @Override
    public void addOutputStream(OutputStream out) {
        // TODO Auto-generated method stub
    }

    @Override
    public void removeOutputStream(OutputStream out) {
        // TODO Auto-generated method stub
    }
}
